﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>

#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_InputNumberWindow.h"

namespace
{

uint64_t CalcDigitUnit(int base, int index) NN_NOEXCEPT
{
    uint64_t unit = 1;
    for (int i = 0; i < index; i++)
    {
        unit *= base;
    }

    return unit;
}

}  // anonymous

namespace nfpdebug { namespace ui {

void InputNumberWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    auto updateDigit = [this](int index, int amount)
    {
        int base = (m_Base == NumberBase::Hex ? 16 : 10);
        uint64_t unit = CalcDigitUnit(base, m_Digits - index - 1);
        uint64_t digit = m_Number / unit % base;

        m_Number -= digit * unit;
        digit = (digit + amount + base) % base;
        m_Number += digit * unit;
    };

    if (controller.IsRepeated(nn::hid::NpadButton::Right::Mask))
    {
        m_CursorIndex = (m_CursorIndex + 1) % m_Digits;
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Left::Mask))
    {
        m_CursorIndex = (m_CursorIndex + m_Digits - 1) % m_Digits;
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        updateDigit(m_CursorIndex, 1);
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        updateDigit(m_CursorIndex, -1);
    }

    if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        if (m_DecideHandler != nullptr)
        {
            m_DecideHandler(controller.GetNpadId(), m_Number);
        }
    }

    Window::UpdateInput(controller);
}

void InputNumberWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    renderer.SetTextScale(1.2f, 1.2f);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    float x = rect.x + 6;
    float y = rect.y + 4;
    if (m_Text != nullptr)
    {
        renderer.DrawText(x, y, "%s", m_Text);
    }
    RenderDigits(x + 8, y + renderer.GetTextLineHeight() * 2);

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void InputNumberWindow::RenderDigits(float x, float y) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();
    renderer.SetFixedMode(true, FixedModeTextWidth);

    auto getDigit = [this](int index)
    {
        int base = (m_Base == NumberBase::Hex ? 16 : 10);
        uint64_t unit = CalcDigitUnit(base, m_Digits - index - 1);

        return static_cast<int>(m_Number / unit % base);
    };

    float dx = x;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
    renderer.DrawText(dx, y - 2, "<");
    dx += FixedModeTextWidth + 12;

    for (int i = 0; i < m_Digits; i++)
    {
        if (i == m_CursorIndex)
        {
            // 上下アロー
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
            renderer.DrawText(dx, y - renderer.GetTextLineHeight() * 0.8f, "^");
            renderer.DrawText(dx, y + renderer.GetTextLineHeight() * 0.8f, "v");

            DrawCursor(
                dx,
                y,
                FixedModeTextWidth + 4,
                renderer.GetTextLineHeight() - 8,
                GetCursorFadeColor());
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::Turquoise));
        }
        else
        {
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
        }

        int digit = getDigit(i);
        renderer.DrawText(dx, y, "%X", digit);
        dx += FixedModeTextWidth + 4;
    }

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
    renderer.DrawText(dx + 8, y - 2, ">");
    renderer.SetFixedMode(false);
}

}}  // nfpdebug::ui
