﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nfp/nfp_DebugTypes.h>

#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_SelectableWindow.h"

namespace nfpdebug { namespace ui {

enum AdminWindowItem : int
{
#ifdef NFPDEBUG_SYSTEM
    AdminWindowItem_NxApplicationId,
    AdminWindowItem_ClassicApplicationId,
#else
    // 一般公開用機能は選択範囲外に定義する
    AdminWindowItem_NxApplicationId = -2,
    AdminWindowItem_ClassicApplicationId,
#endif  // ifndef NFPDEBUG_SYSTEM

    AdminWindowItem_AccessId,

    AdminWindowItem_Count
};

/**
 * @brief   AdminInfo を編集するウィンドウです。
 */
class EditAdminWindow : public SelectableWindow
{
public:
    EditAdminWindow() NN_NOEXCEPT :
        m_AdminInfo(),
        m_CursorIndex(0),
        m_DecideHandler(nullptr)
    {
        SetCursorVisible(true);
        SetSize(560, ItemCountMax * 32 + 32);
    }

    /**
     * @brief   カーソル位置を設定します。
     */
    virtual void SetCursorIndex(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        m_CursorIndex = std::min(std::max(index, 0), ItemCountMax - 1);
    }

    /**
     * @brief   カーソル位置を取得します。
     */
    virtual int GetCursorIndex() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_CursorIndex;
    }

    /**
     * @brief   編集対象の AdminInfo を設定します。
     */
    void SetAdminInfo(const nn::nfp::AdminInfoDebug& info) NN_NOEXCEPT
    {
        m_AdminInfo = info;
    }

    /**
     * @brief   決定時のコールバック関数を登録します。
     */
    void SetDecideHandler(WindowHandlerTypeWithIndex handler) NN_NOEXCEPT
    {
        m_DecideHandler = handler;
    }

    /**
     * @brief   カーソル位置の項目の Y 座標を取得します。
     */
    virtual float GetSelectedItemY() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetY() + GetDefaultLineHeight() * m_CursorIndex + GetHeaderHeight();
    }

    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    virtual float GetRenderHeight() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetDefaultLineHeight() * ItemCountMax + GetHeaderHeight() + 8;
    }

private:
    static const int ItemCountMax;

private:
    void DrawCurrentInfo() NN_NOEXCEPT;

private:
    nn::nfp::AdminInfoDebug     m_AdminInfo;        //!< 編集対象の AdminInfo
    int                         m_CursorIndex;      //!< カーソル位置
    WindowHandlerTypeWithIndex  m_DecideHandler;    //!< 決定時のコールバック
};

}}  // nfpdebug::ui
