﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nfp/nfp_DebugTypes.h>

#include "../NfpDebugTool_Util.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_EditAdminWindow.h"

namespace nfpdebug { namespace ui {

const int EditAdminWindow::ItemCountMax = AdminWindowItem_Count;

void EditAdminWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        m_CursorIndex = (m_CursorIndex + ItemCountMax - 1) % ItemCountMax;
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        m_CursorIndex = (m_CursorIndex + 1) % ItemCountMax;
    }
    else if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        if (m_DecideHandler != nullptr)
        {
            m_DecideHandler(controller.GetNpadId(), 0, m_CursorIndex);
        }
    }

    Window::UpdateInput(controller);
}

void EditAdminWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    renderer.SetTextScale(1.2f, 1.2f);

    DrawCurrentInfo();

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void EditAdminWindow::DrawCurrentInfo() NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float dx = rect.x + 12;
    float dy = rect.y + 2;

    DrawCursor(
        dx - 8,
        dy + m_CursorIndex * renderer.GetTextLineHeight(),
        rect.width - 4,
        renderer.GetTextLineHeight() - 8,
        GetCursorFadeColor());

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    float offsetX = 256;

#ifdef NFPDEBUG_SYSTEM
    auto applicationId = GetActualApplicationId(
        m_AdminInfo.applicationId,
        m_AdminInfo.applicationIdExt);
    renderer.DrawText(dx, dy, "Application ID (NX)");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy,
        "0x%08X%08X",
        static_cast<uint32_t>((applicationId >> 32) & 0xFFFFFFFF),
        static_cast<uint32_t>(applicationId & 0xFFFFFFFF));
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Title ID (CTR / Cafe)");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy,
        "0x%08X%08X",
        static_cast<uint32_t>((m_AdminInfo.applicationId.value >> 32) & 0xFFFFFFFF),
        static_cast<uint32_t>(m_AdminInfo.applicationId.value & 0xFFFFFFFF));
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();
#endif  // ifdef NFPDEBUG_SYSTEM

#if 0
    renderer.DrawText(dx, dy, "Ext application ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%02X", m_AdminInfo.applicationIdExt);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();
#endif

    renderer.DrawText(dx, dy, "Access ID");
    renderer.SetFixedMode(true, FixedModeTextWidth);
    renderer.DrawText(dx + offsetX, dy, "0x%08X", m_AdminInfo.accessId);
    renderer.SetFixedMode(false);
    dy += renderer.GetTextLineHeight();

}

}}  // nfpdebug::ui
