﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "NfpDebugTool_DeviceListWindow.h"
#include "../NfpDebugTool_Util.h"

namespace nfpdebug { namespace ui {

const nn::hid::NpadIdType DeviceListWindow::InvalidNpadId =
    static_cast<nn::hid::NpadIdType>(0x39393939);

namespace
{

/**
 * @brief   インジケータの色を取得
 */
void GetIndicatorColor(graphics::Color* pOutColor, bool isOn) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutColor);

    *pOutColor = isOn ? graphics::Colors::Green : graphics::Colors::Gray;
}

}  // anonymous

void DeviceListWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    renderer.SetTextScale(1.2f, 1.2f);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);
    PrintDeviceList(rect.x + 4, rect.y + 2, rect.width - 8, rect.height - 4);

    renderer.SetTextScale(prevScale.x, prevScale.y);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
}

void DeviceListWindow::PrintDeviceList(float x, float y, float width, float height) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    bool isDeviceConnected = false;
    for (int i = 0; i < npad::NpadIdCountMax; i++)
    {
        auto id = npad::NpadIds[i];

        const auto* styleName = GetNpadStyleName(id);
        if (styleName == nullptr)
        {
            continue;
        }

        isDeviceConnected = true;

        if (m_ActiveNpadId == id)
        {
            DrawCursor(dx, dy, width, renderer.GetTextLineHeight() - 8, graphics::Colors::Crimson);
        }

        PrintIndicator(dx, dy, i);
        renderer.DrawText(dx + 56, dy, "%s", styleName);
        dy += DisplayLineOffset;
    }

    if (!isDeviceConnected)
    {
        renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
        renderer.DrawText(dx, dy, "No device connected\n");
        renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
    }
}

void DeviceListWindow::PrintIndicator(float x, float y, int idIndex) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    renderer.SetTextScale(0.6f, 0.8f);

    for (int i = 0; i < 4; i++)
    {
        uint8_t mask = 0x01 << i;
        graphics::Color color;
        GetIndicatorColor(&color, npad::IndicatorPattern[idIndex] & mask);
        renderer.SetTextColor(GetDisplayColor(color));
        renderer.DrawText(x + i * 12, y + 4, "■");
    }

    renderer.SetTextScale(1.2f, 1.2f);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
}

}}  // nfpdebug::ui
