﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nfp/nfp_NpadApi.h>

#include "../NfpDebugTool_Util.h"
#include "NfpDebugTool_CabinetResultWindow.h"

namespace nfpdebug { namespace ui {

void CabinetResultWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    Window::UpdateInput(controller);

    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        auto handler = GetCancelHandler();
        if (handler != nullptr)
        {
            handler(controller.GetNpadId(), 0);
        }
    }
}

void CabinetResultWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);
    renderer.SetTextScale(1.2f, 1.2f);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float dx = rect.x + 6;
    float dy = rect.y + 6;
    float offset = 220;

    if (m_CabinetResult.HasDeviceHandle())
    {
        DrawDeviceHandle(rect.x + rect.width - 380, dy, offset, m_CabinetResult.deviceHandle);
    }

    if (m_CabinetResult.HasTagInfo())
    {
        dy += DrawTagInfo(dx, dy, offset, m_CabinetResult.tagInfo) + 8;
    }

    if (m_CabinetResult.HasRegisterInfo())
    {
        dy += DrawRegisterInfo(dx, dy, offset, m_CabinetResult.registerInfo) + 8;
    }

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

float CabinetResultWindow::DrawDeviceHandle(
    float x,
    float y,
    float offsetX,
    const nn::nfp::DeviceHandle& handle) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    float dx = x;
    float dy = y;

    dy += DrawInfoTitle(dx, dy, "Device info");
    dx += 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    nn::hid::NpadIdType npadId;
    auto result = nn::nfp::GetNpadId(&npadId, handle);
    const char* deviceName = result.IsSuccess() ? GetNpadIdName(npadId) : "???";

    renderer.DrawText(dx, dy, "Used device");
    renderer.DrawText(dx + offsetX, dy, "%s", deviceName);
    dy += renderer.GetTextLineHeight();

    return dy - y;
}

}}  // nfpdebug::ui
