﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Result.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/mii/mii_StoreDataContext.h>
#include <nn/mii/mii_Nfp.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/nfp/nfp_DebugTypes.h>
#include <nn/nfp/nfp_Result.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ResEndian.h>
#include <nn/util/util_ScopeExit.h>

#include "graphics/NfpDebugTool_Renderer.h"
#include "npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_Util.h"

namespace {

const struct
{
    nn::hid::NpadIdType     npadId;
    const char*             name;
} NpadNameList[] =
{
    { nn::hid::NpadId::No1,      "No.1" },
    { nn::hid::NpadId::No2,      "No.2" },
    { nn::hid::NpadId::No3,      "No.3" },
    { nn::hid::NpadId::No4,      "No.4" },
    { nn::hid::NpadId::No5,      "No.5" },
    { nn::hid::NpadId::No6,      "No.6" },
    { nn::hid::NpadId::No7,      "No.7" },
    { nn::hid::NpadId::No8,      "No.8" },
    { nn::hid::NpadId::Handheld, "Handheld" },
};

const nn::Bit64 NxTitleIdExtentionMask = 0x00000000F0000000ull;

template <typename T>
void SetValueToTag(void* pDst, T value) NN_NOEXCEPT
{
    T tagByteOrderValue;
    nn::util::StoreBigEndian<T>(&tagByteOrderValue, value);
    std::memcpy(pDst, &tagByteOrderValue, sizeof(T));
}

nn::Bit32 CalculateCrc32(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    const nn::Bit8* c = reinterpret_cast<const nn::Bit8*>(pData);
    const nn::Bit32 CrcPoly2 = 0xEDB88320;
    const size_t CharBit = 8;
    nn::Bit32 r;
    size_t i, j;

    r = 0xFFFFFFFF;
    for (i = 0; i < dataSize; i++)
    {
        r ^= c[i];

        for (j = 0; j < CharBit; j++)
        {
            if (r & 1)
            {
                r = (r >> 1) ^ CrcPoly2;
            }
            else
            {
                r >>= 1;
            }
        }
    }
    return r ^ 0xFFFFFFFF;
}

}  // anonymous

namespace nfpdebug {

const float DisplayLineOffset   = 30;
const float DisplayColumnOffset = 60;

const char* GetNpadIdName(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    for (const auto& nameset : NpadNameList)
    {
        if (nameset.npadId == npadId)
        {
            return nameset.name;
        }
    }

    return "???";
}

const char* GetNpadStyleName(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(npadId);

    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        return "FullKeyController";
    }
    else if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        return "JoyCon L/R";
    }
    else if (style.Test<nn::hid::NpadStyleJoyLeft>())
    {
        return "JoyCon Left";
    }
    else if (style.Test<nn::hid::NpadStyleJoyRight>())
    {
        return "JoyCon Right";
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        return "Handheld";
    }

    return nullptr;
}

const char* GetTagProtocolName(nn::nfp::NfcProtocol protocol) NN_NOEXCEPT
{
    switch (protocol)
    {
    case nn::nfp::NfcProtocol_TypeA:     return "NFC-A";
    case nn::nfp::NfcProtocol_TypeB:     return "NFC-B";
    case nn::nfp::NfcProtocol_TypeF:     return "NFC-F";
    case nn::nfp::NfcProtocol_Type15693: return "ISO-15693";
    default: return "Unknown";
    }
}

const char* GetTagTypeName(nn::nfp::TagType tagType) NN_NOEXCEPT
{
    switch (tagType)
    {
    case nn::nfp::TagType_Type1:    return "TYPE1";
    case nn::nfp::TagType_Type2:    return "TYPE2";
    case nn::nfp::TagType_Type3:    return "TYPE3";
    case nn::nfp::TagType_Type4A:   return "TYPE4A";
    case nn::nfp::TagType_Type4B:   return "TYPE4B";
    case nn::nfp::TagType_Iso15693: return "ISO-15693";
    case nn::nfp::TagType_Mifare:   return "MIFARE";
    default: return "Unknown";
    }
}

const char* GetFontRegionName(nn::nfp::FontRegion fontRegion) NN_NOEXCEPT
{
    switch (fontRegion)
    {
    case nn::nfp::FontRegion_JpUsEu: return "JpUsEu";
    case nn::nfp::FontRegion_China:  return "China";
    case nn::nfp::FontRegion_Korea:  return "Korea";
    case nn::nfp::FontRegion_Taiwan: return "Taiwan";
    default: return "Unknown";
    }
}

void PrintHeader(graphics::Renderer* pRenderer, const char* title) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pRenderer);
    NN_ASSERT_NOT_NULL(title);

    pRenderer->SetTextScale(2.0f, 2.0f);
    pRenderer->SetTextColor(graphics::Colors::White);
    pRenderer->DrawText(20, 20, "%s", title);
    pRenderer->SetTextScale(1.2f, 1.2f);

    pRenderer->DrawHorizontalLine(0, 80, 1280, graphics::Colors::Silver, 2.0f);
}

bool IsSameTag(const nn::nfp::TagId& tagId1, const nn::nfp::TagId& tagId2) NN_NOEXCEPT
{
    if (tagId1.length != tagId2.length)
    {
        return false;
    }

    return std::memcmp(tagId1.uid, tagId2.uid, tagId1.length) == 0;
};

nn::Bit64 GetActualApplicationId(
    const nn::ncm::ApplicationId& applicationId,
    nn::Bit8 applicationIdExt) NN_NOEXCEPT
{
    nn::Bit64 uniqueId = (static_cast<nn::Bit64>(applicationIdExt & 0x0F) << 28);

    auto actualId = applicationId.value;
    actualId &= ~NxTitleIdExtentionMask;
    actualId |= uniqueId;

    return actualId;
}

void GetApplicationIdForAmiibo(
    nn::ncm::ApplicationId* pOutApplicationId,
    nn::Bit8* pOutApplicationIdExt,
    nn::Bit64 actualApplicationId) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutApplicationId);
    NN_ASSERT_NOT_NULL(pOutApplicationIdExt);

    *pOutApplicationIdExt = static_cast<nn::Bit8>((actualApplicationId >> 28) & 0x0F);

    nn::Bit64 compatibleId;
    compatibleId = actualApplicationId;
    compatibleId &= ~NxTitleIdExtentionMask;
    compatibleId |= (static_cast<nn::Bit64>(0x3) << 28);

    pOutApplicationId->value = compatibleId;
}

nn::Bit32 GetExtentionCrc(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    struct NfpFormat
    {
        nn::Bit8 miiData[96];
        nn::Bit8 titleId[8];
        nn::Bit8 writeCounter[2];
        nn::Bit8 accessId[4];
        nn::Bit8 titleIdEx[1];
        nn::Bit8 miiExtVersion[1];
        nn::Bit8 miiExtension[8];
        nn::Bit8 reserved[20];
    } nfpFormat;

    std::memcpy(nfpFormat.miiData, &nfpData.registerInfo.miiDataCore, sizeof(nfpFormat.miiData));
    SetValueToTag(nfpFormat.titleId, nfpData.adminInfo.applicationId.value);
    SetValueToTag(nfpFormat.writeCounter, nfpData.commonInfo.writeCounter);
    SetValueToTag(nfpFormat.accessId, nfpData.adminInfo.accessId);
    SetValueToTag(nfpFormat.titleIdEx, nfpData.adminInfo.applicationIdExt);
    SetValueToTag(nfpFormat.miiExtVersion, nfpData.registerInfo.miiExtentionVersion);
    std::memcpy(nfpFormat.miiExtension, &nfpData.registerInfo.miiDataExtention, sizeof(nfpFormat.miiExtension));
    std::memcpy(nfpFormat.reserved, nfpData.registerInfo.reserved, sizeof(nfpFormat.reserved));

    return CalculateCrc32(&nfpFormat, sizeof(nfpFormat));
}

nn::Result ConvertStoreDataToNfpMiiData(
    nn::mii::NfpStoreDataCore* pOutCore,
    nn::mii::NfpStoreDataExtention* pOutExtention,
    const nn::mii::StoreData& storeData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutCore);
    NN_ASSERT_NOT_NULL(pOutExtention);

    nn::mii::NfpStoreData nfpStoreData;
    nn::mii::StoreDataContext context;
    NN_RESULT_TRY(nn::mii::BuildToNfpStoreData(&nfpStoreData, &context, storeData))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(nn::nfp::ResultInvalidMii());
        }
    NN_RESULT_END_TRY

    std::memcpy(pOutCore, &nfpStoreData.core, sizeof(nfpStoreData.core));
    std::memcpy(pOutExtention, &nfpStoreData.extention, sizeof(nfpStoreData.extention));

    NN_RESULT_SUCCESS;
}

void CreateSampleMiiData(nn::mii::StoreData* pOutMiiData, int type) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutMiiData);

    nn::mii::StoreDataContext context;
    nn::mii::StoreDataAccessor accessor(pOutMiiData);
    accessor.BuildDefault(&context, 0);

    switch (type)
    {
    case 0:
        {
            // ニックネーム
            const char16_t* nicknameUtf16 = u"Nickname";
            nn::mii::Nickname nickname = {};
            nickname.Set(reinterpret_cast<const uint16_t*>(nicknameUtf16));
            accessor.SetNickname(nickname, nn::mii::FontRegion_JpUsEu);

            // お気に入りの色 : 黄色
            // 性別 : 男性
            // 身長 : 最も高い
            // 体格 : 最も細い
            accessor.SetFavoriteColor(nn::mii::FavoriteColor_Yellow);
            accessor.SetGender(nn::mii::Gender_Male);
            accessor.SetHeight(nn::mii::HeightMax);
            accessor.SetBuild(nn::mii::BuildMin);

            // 拡張部分 : 髪の色
            accessor.SetHairColor(50);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

const char* GetResultText(nn::Result result) NN_NOEXCEPT
{
    // ひとまず Mount 関連のみ
    if (nn::nfp::ResultNfcDeviceNotFound::Includes(result))
    {
        return "ResultNfcDeviceNotFound";
    }
    else if (nn::nfp::ResultNeedRetry::Includes(result))
    {
        return "ResultNeedRetry";
    }
    else if (nn::nfp::ResultNeedRestart::Includes(result))
    {
        return "ResultNeedRestart";
    }
    else if (nn::nfp::ResultNfcDisabled::Includes(result))
    {
        return "ResultNfcDisabled";
    }
    else if (nn::nfp::ResultNeedRestore::Includes(result))
    {
        return "ResultNeedRestore";
    }
    else if (nn::nfp::ResultNeedFormat::Includes(result))
    {
        return "ResultNeedFormat";
    }
    else if (nn::nfp::ResultNotSupported::Includes(result))
    {
        return "ResultNotSupported";
    }
    else if (nn::nfp::ResultInvalidFormatVersion::Includes(result))
    {
        return "ResultNotInvalidFormatVersion";
    }
    else
    {
        return "-";
    }
}

}  // nfpdebug
