﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_PrivateApi.h>
#include <nn/nfp/nfp_DebugApi.h>

#include "npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_Util.h"

namespace nfpdebug {

/**
 * @brief   NFP API の処理結果です。
 */
enum class NfpResult
{
    Success,            //!< 成功
    Failed,             //!< 失敗 (汎用)
    DeviceLost,         //!< デバイス喪失
    TagLost,            //!< タグ喪失
    UnsupportedTag,     //!< 非対応タグ
    NotForeground,      //!< BG にいる
    NotBroken           //!< 復旧不要
};

/**
 * @brief   削除する情報の種類です。
 */
enum DeleteInfoType
{
    DeleteInfoType_ApplicationArea,     //!< アプリケーション領域
    DeleteInfoType_RegisterInfo         //!< 登録情報
};

/**
 * @brief   検出した amiibo に関する情報です。
 */
struct NfpInfo
{
    bool                    isValidTag;         //!< 有効なタグか
    bool                    isAmiibo;           //!< amiibo か
    bool                    hasValidRam;        //!< RAM 領域が正常
    bool                    hasRegisterInfo;    //!< 登録済み

    nn::nfp::TagInfo        tagInfo;            //!< タグ情報
    nn::nfp::ModelInfo      modelInfo;          //!< amiibo の型式情報
    nn::nfp::RegisterInfo   publicRegisterInfo; //!< 公開登録情報 (allData 編集後は齟齬が発生するので参照してはいけない)
    nn::nfp::NfpData        allData;            //!< タグの全データ

    /**
     * @brief   保持している情報をクリアします。
     */
    void Clear() NN_NOEXCEPT
    {
        isValidTag      = false;
        isAmiibo        = false;
        hasValidRam     = false;
        hasRegisterInfo = false;
        std::memset(&tagInfo,   0, sizeof(tagInfo));
        std::memset(&modelInfo, 0, sizeof(modelInfo));
        std::memset(&allData,   0, sizeof(allData));
    }

    /**
     * @brief   有効なタグかどうかを返します。
     */
    bool IsValidTag() const NN_NOEXCEPT
    {
        return isValidTag;
    }

    /**
     * @brief   amiibo かどうかを返します。
     */
    bool IsAmiibo() const NN_NOEXCEPT
    {
        return isAmiibo;
    }

    /**
     * @brief   RAM 領域が正常かどうか返します。
     */
    bool HasValidRamArea() const NN_NOEXCEPT
    {
        return hasValidRam;
    }

    /**
     * @brief   アプリケーション専用領域の存在フラグを設定します。
     */
    void EnableApplicationArea() NN_NOEXCEPT
    {
        auto mask = nn::nfp::RegisterInfoFlags_ApplicationArea;
        allData.adminInfo.registerInfo |= mask;
    }

    /**
     * @brief   アプリケーション専用領域が存在するか返します。
     */
    bool HasApplicationArea() const NN_NOEXCEPT
    {
        auto mask = nn::nfp::RegisterInfoFlags_ApplicationArea;
        return (allData.adminInfo.registerInfo & mask) == mask;
    }

    /**
     * @brief   登録情報の存在フラグを設定します。
     */
    void EnableRegisterInfo() NN_NOEXCEPT
    {
        auto mask = nn::nfp::RegisterInfoFlags_CommonArea;
        allData.adminInfo.registerInfo |= mask;
        hasRegisterInfo = true;
    }

    /**
     * @brief   登録情報が存在するか返します。
     */
    bool HasRegisterInfo() const NN_NOEXCEPT
    {
        return hasRegisterInfo;
    }
};

// コールバック型
typedef void(*NfpTagDetectCallbackType)(void*);
typedef void(*NfpTagLostCallbackType)(void*);
typedef void(*NfpCancelCallbackType)(void*);
typedef void(*NfpResultCallbackType)(NfpResult, void*);

/**
 * @brief   NFP に関する処理を行うクラスです。
 */
class NfpProcessor final
{
    NN_DISALLOW_COPY(NfpProcessor);
    NN_DISALLOW_MOVE(NfpProcessor);

public:
    NfpProcessor() NN_NOEXCEPT :
        m_Mutex(true),
        m_IsRunning(false),
        m_IsActivated(false),
        m_Thread(),
        m_pThreadStack(nullptr),
        m_DetectEvent(),
        m_LostEvent(),
        m_SuspendEvent(),
        m_ResumeEvent(),
        m_DeviceHandle(),
        m_NfpInfo(),
        m_NextCommand(Command::None),
        m_DetectCallback(nullptr),
        m_pDetectCallbackArgument(nullptr),
        m_LostCallback(nullptr),
        m_pLostCallbackArgument(nullptr),
        m_CancelCallback(nullptr),
        m_pCancelCallbackArgument(nullptr),
        m_FinishCallback(nullptr),
        m_pFinishCallbackArgument(nullptr)
    {}

    ~NfpProcessor() NN_NOEXCEPT
    {
        Shutdown();
    }

    void Setup(char* pThreadStack, size_t stackSize) NN_NOEXCEPT;
    void Shutdown() NN_NOEXCEPT;

    bool Activate(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    void Suspend() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;

    void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT;

    void SetNfpInfo(const NfpInfo& info) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_NfpInfo = info;
    }

    void GetNfpInfo(NfpInfo* pOutInfo) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutInfo);

        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        *pOutInfo = m_NfpInfo;
    }

    void SetSystemInfo(const nn::nfp::SystemInfo& info) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_NfpInfo.allData.systemInfo = info;
        UpdateExtentionCrc();
    }

    void SetRegisterInfo(const nn::nfp::RegisterInfoDebug& info) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_NfpInfo.allData.registerInfo = info;
        m_NfpInfo.hasRegisterInfo      = true;
        m_NfpInfo.EnableRegisterInfo();
        UpdateExtentionCrc();
    }

    void SetAdminInfo(const nn::nfp::AdminInfoDebug& info) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_NfpInfo.allData.adminInfo = info;
        UpdateExtentionCrc();
    }

    void SetApplicationArea(const char* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pData);
        NN_ASSERT_LESS_EQUAL(dataSize, nn::nfp::ApplicationAreaSizeV2);

        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        std::memcpy(m_NfpInfo.allData.applicationArea, pData, dataSize);
        m_NfpInfo.EnableApplicationArea();
    }

    void SetDetectCallback(NfpTagDetectCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_DetectCallback = callback;
        m_pDetectCallbackArgument = pArgument;
    }

    void SetLostCallback(NfpTagLostCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_LostCallback = callback;
        m_pLostCallbackArgument = pArgument;
    }

    void SetCancelCallback(NfpCancelCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_CancelCallback = callback;
        m_pCancelCallbackArgument = pArgument;
    }

    void SetFinishCallback(NfpResultCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_FinishCallback = callback;
        m_pFinishCallbackArgument = pArgument;
    }

    void ClearCallback() NN_NOEXCEPT
    {
        m_DetectCallback = nullptr;
        m_LostCallback   = nullptr;
        m_CancelCallback = nullptr;
        m_FinishCallback = nullptr;
    }

    void NotifyLostFocus() NN_NOEXCEPT;

    // タグ検出系
    nn::Result StartDetection() NN_NOEXCEPT;
    nn::Result StopDetection() NN_NOEXCEPT;

    // 書き込み系
    nn::Result Flush() NN_NOEXCEPT;
    nn::Result FlushDebug() NN_NOEXCEPT;
    nn::Result Format() NN_NOEXCEPT;
    nn::Result Restore() NN_NOEXCEPT;
    nn::Result DeleteInfo(DeleteInfoType infoType) NN_NOEXCEPT;
    nn::Result BreakTag(nn::nfp::BreakType breakType) NN_NOEXCEPT;

    // cabinet 系
    nn::Result CallNicknameAndOwner(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        nn::nfp::TagInfo* pOutTagInfo,
        bool* pOutIsRegistered,
        nn::nfp::RegisterInfo* pOutRegisterInfo,
        bool isNicknameRequired) NN_NOEXCEPT;
    nn::Result CallNicknameAndOwner(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        bool* pOutIsRegistered,
        nn::nfp::RegisterInfo* pOutRegisterInfo,
        bool isNicknameRequired,
        const nn::nfp::TagInfo& tagInfo) NN_NOEXCEPT;
    nn::Result CallNicknameAndOwner(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        bool* pOutIsRegistered,
        nn::nfp::RegisterInfo* pOutRegisterInfo,
        bool isNicknameRequired,
        const nn::nfp::TagInfo& tagInfo,
        const nn::nfp::RegisterInfo& registerInfo) NN_NOEXCEPT;
    nn::Result CallGameDataEraser(
        nn::nfp::DeviceHandle* pOutDeviceHandle) NN_NOEXCEPT;
    nn::Result CallGameDataEraser(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT;
    nn::Result CallRestorer(
        nn::nfp::DeviceHandle* pOutDeviceHandle) NN_NOEXCEPT;
    nn::Result CallRestorer(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT;
    nn::Result CallFormatter(
        nn::nfp::DeviceHandle* pOutDeviceHandle,
        nn::nfp::TagInfo* pOutTagInfo) NN_NOEXCEPT;

private:
    enum class Command
    {
        None,
        Flush,
        FlushDebug,
        Format,
        Restore,
        DeleteApplicationArea,
        DeleteRegisterInfo,
        BreakActivation,
        BreakHmac,
        Suspend
    };

private:
    static void ThreadFunction(void* pArg) NN_NOEXCEPT;

    void UpdateExtentionCrc() NN_NOEXCEPT
    {
        auto& nfpData = m_NfpInfo.allData;
        nfpData.registerInfo.extentionCrc = GetExtentionCrc(nfpData);
    }

    void ThreadFunctionImpl() NN_NOEXCEPT;
    void CheckEvent() NN_NOEXCEPT;
    bool HandleCommand() NN_NOEXCEPT;
    void ReadAmiibo() NN_NOEXCEPT;
    nn::Result HandleCommonResult(nn::Result result) NN_NOEXCEPT;
    nn::Result HandleFlushResult(nn::Result result) NN_NOEXCEPT;

private:
    mutable nn::os::Mutex       m_Mutex;

    bool                        m_IsRunning;
    bool                        m_IsActivated;

    nn::os::ThreadType          m_Thread;
    char*                       m_pThreadStack;
    nn::os::SystemEventType     m_DetectEvent;
    nn::os::SystemEventType     m_LostEvent;
    nn::os::EventType           m_SuspendEvent;
    nn::os::EventType           m_ResumeEvent;
    nn::nfp::DeviceHandle       m_DeviceHandle;

    NfpInfo                     m_NfpInfo;
    Command                     m_NextCommand;

    NfpTagDetectCallbackType    m_DetectCallback;
    void*                       m_pDetectCallbackArgument;
    NfpTagLostCallbackType      m_LostCallback;
    void*                       m_pLostCallbackArgument;
    NfpCancelCallbackType       m_CancelCallback;
    void*                       m_pCancelCallbackArgument;
    NfpResultCallbackType       m_FinishCallback;
    void*                       m_pFinishCallbackArgument;
};

}  // nfpdebug
