﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/fs.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <glv_ScissorBoxView.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/util/util_FormatString.h>

#include "NandVerifier_HashList.h"
#include "NandVerifier_Viewer.h"
#include "NandVerifier_VerifyScene.h"
#include "NandVerifier_PartitionDefinitions.h"

namespace
{
    bool GetLoadSource(const char* path) NN_NOEXCEPT
    {
        if (std::string(path).substr(0, 3) == "rom")
        {
            return false;
        }

        return true;
    }
}

VerifyScene::VerifyScene(HashList** ppInHashList) NN_NOEXCEPT
    : m_ppHashList(ppInHashList)
{
    auto table = new glv::Table(
        "ppp.q,"
        "p---.,"
        "-p--."
    );

    m_Viewer = new Viewer(ppInHashList);
    m_Status = new glv::Label("\n", glv::Label::Spec(glv::Place::TR, 5, 0, 35));

    m_Viewer->Initialize([&](std::string str) { Verify(str); });

    auto dumpButton = new CommandButton("Dump", [&] { Dump(); });
    auto loadButton = new CommandButton("Load", [&] { Load(); });
    auto settingsButton = new CommandButton("Settings", [&] { Settings(); });
    auto exitButton = new CommandButton("Exit", [&] { std::quick_exit(0); });

    *table << loadButton
        << dumpButton
        << settingsButton
        << exitButton
        << m_Status;

    *table << m_Viewer;

    table->fit(false);
    table->arrange();

    *this << table;
}

nn::Result VerifyScene::LoadAnswerFileFromRom(const char* path) NN_NOEXCEPT
{
    const char* MountName = "rom";
    size_t cacheSize = 0;
    NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&cacheSize));

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_RESULT_DO(nn::fs::MountRom(MountName, cacheBuffer.get(), cacheSize));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };

    NN_RESULT_DO(m_Viewer->ReadAnswerFile(path));

    NN_RESULT_SUCCESS;
}

nn::Result VerifyScene::LoadAnswerFileFromSdCard(const char* path) NN_NOEXCEPT
{
    const char* MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCardForDebug(MountName));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };

    NN_RESULT_DO(m_Viewer->ReadAnswerFile(path));

    NN_RESULT_SUCCESS;
}

nn::Result VerifyScene::LoadAnswerFileImpl(const char* path) NN_NOEXCEPT
{
    if (GetLoadSource(path))
    {
        NN_RESULT_DO(LoadAnswerFileFromSdCard(path));
    }
    else
    {
        NN_RESULT_DO(LoadAnswerFileFromRom(path));
    }

    NN_RESULT_SUCCESS;
}

void VerifyScene::Verify(std::string path) NN_NOEXCEPT
{
    //パスの形式ではない場合はスキップする
    if (path.find(":/") == std::string::npos)
    {
        return;
    }

    const int BufferSize = 200;
    char buffer[BufferSize];

    NN_LOG("Path: %s\n", path.c_str());
    if (LoadAnswerFileImpl(path.c_str()).IsFailure())
    {
        m_Status->setValue("Load AnswerFile Failed.");
        return;
    }

    if (m_Viewer->Verify())
    {
        nn::util::SNPrintf(buffer, BufferSize, "Loaded: %s\nResult: Success\n", path.c_str());
    }
    else
    {
        nn::util::SNPrintf(buffer, BufferSize, "Loaded: %s\nResult: Failed\n", path.c_str());
    }

    m_Status->setValue(std::string(buffer));
    m_Viewer->DumpVerifyResult(m_Status);

    m_Viewer->Refresh();
    m_Viewer->ShowVerifyResult();
}

nn::Result VerifyScene::LoadFileListFromRom() NN_NOEXCEPT
{
    size_t cacheSize = 0;
    NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&cacheSize));

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);

    NN_RESULT_DO(nn::fs::MountRom("rom", cacheBuffer.get(), cacheSize));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    m_Viewer->ReadFileList("rom:/");
    m_Viewer->Refresh();
    m_Viewer->ShowVerifyResult();

    NN_RESULT_SUCCESS;
}

nn::Result VerifyScene::LoadFileListFromSdCard() NN_NOEXCEPT
{
    const char* MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCardForDebug(MountName));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };


    m_Viewer->ReadFileList("SdCard:/");
    m_Viewer->Refresh();
    m_Viewer->ShowVerifyResult();

    NN_RESULT_SUCCESS;
}

nn::Result VerifyScene::LoadFileListImpl() NN_NOEXCEPT
{
    m_Viewer->ClearAnswerFileList();
    LoadFileListFromSdCard();
    LoadFileListFromRom();

    NN_RESULT_SUCCESS;
}

void VerifyScene::Load() NN_NOEXCEPT
{
    nn::Result result = LoadFileListImpl();
    if (result.IsSuccess())
    {
        m_Status->setValue("Success");
    }
    else
    {
        const int BufferSize = 300;
        char buffer[BufferSize];

        nn::util::SNPrintf(buffer, BufferSize, "Failed. result = %08x\n", result.GetInnerValueForDebug());
        m_Status->setValue(std::string(buffer));
    }
}

void VerifyScene::Settings() NN_NOEXCEPT
{
    m_Viewer->Refresh();
    m_Viewer->Settings();
}

nn::Result VerifyScene::DumpImpl(std::string& str) NN_NOEXCEPT
{
    std::string merged;

    for (int i = 0; i < PartitionNum; i++)
    {
        const int BufferSize = 300;
        char buffer[BufferSize];

        nn::util::SNPrintf(buffer, BufferSize, "%s %d\n", m_ppHashList[i]->GetPartitionName().c_str(), m_ppHashList[i]->GetStoredSize());
        NN_LOG("%s", buffer);
        m_ppHashList[i]->Sort();

        merged += buffer;

        for (int j = 0; j < m_ppHashList[i]->GetStoredSize(); j++)
        {
            NN_LOG("%s %s\n", m_ppHashList[i]->GetHashValue(j).c_str(), m_ppHashList[i]->GetHashValue(j).c_str());
            merged += m_ppHashList[i]->GetHashValue(j) + " " + m_ppHashList[i]->GetHashValue(j);
            merged += "\n";
        }
    }

    const char* MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCardForDebug(MountName));

    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

    const int PathBufferSize = 200;
    char path[PathBufferSize];

    const int MaxIdValue = 100;
    nn::Result result = nn::fs::CreateDirectory("SdCard:/NandVerifier");
    if (result.IsFailure())
    {
        if (!nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            NN_LOG("Failed.\n");
            return result;
        }
    }

    for (int i = 0; i < MaxIdValue; i++)
    {
        nn::util::SNPrintf(path, PathBufferSize, "%s:/NandVerifier/Dump%04d.ans", MountName, i);
        result = nn::fs::CreateFile(path, 0);
        if (result.IsFailure())
        {
            if (!nn::fs::ResultPathAlreadyExists::Includes(result))
            {
                NN_LOG("Failed.\n");
                return result;
            }
        }
        else
        {
            str = std::string(path);
            break;
        }
    }

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, merged.c_str(), merged.size(), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

void VerifyScene::Dump() NN_NOEXCEPT
{
    const int BufferSize = 300;
    char buffer[BufferSize];
    std::string path;
    nn::Result result = DumpImpl(path);
    if (result.IsSuccess())
    {
        nn::util::SNPrintf(buffer, BufferSize, "Success. %s", path.c_str());
        m_Status->setValue(std::string(buffer));
    }
    else
    {
        nn::util::SNPrintf(buffer, BufferSize, "Failed. result = %08x\n", result.GetInnerValueForDebug());
        m_Status->setValue(std::string(buffer));
    }
}
