﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <glv_ScissorBoxView.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>

#include <nn/util/util_FormatString.h>
#include "NandVerifier_HashGenerator.h"
#include "NandVerifier_LoadScene.h"
#include "NandVerifier_PartitionDefinitions.h"

namespace
{
    const int64_t StorageReadingBufferSize = 8 * 1024 * 1024;
    NN_ALIGNAS(4096) uint8_t s_StorageReadingBuffer[StorageReadingBufferSize];

    const char* MountName = "Contents";

    std::string ConvertHex(void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        std::string ret;
        uint8_t *u8buf = reinterpret_cast<uint8_t*>(buffer);

        const int bufferSize = 3;
        char buf[bufferSize];
        for (size_t i = 0; i < size; i++)
        {
            nn::util::SNPrintf(buf, bufferSize, "%02x", u8buf[i]);
            ret += buf;
        }

        return ret;
    }

    void CalculateContentsId(std::string* pOutContentsId, void* pInHashBuffer, size_t hashSize)
    {
        *pOutContentsId = ConvertHex(pInHashBuffer, hashSize / 2);
    }

    bool IsSaveData(std::string path) NN_NOEXCEPT
    {
        if (path.find("save") != std::string::npos)
        {
            return true;
        }

        return false;
    }

    bool IsPrf2Safe(std::string path) NN_NOEXCEPT
    {
        if (path.find("PRF2SAFE.RCV") != std::string::npos)
        {
            return true;
        }

        return false;
    }

    bool IsLogFile(std::string path) NN_NOEXCEPT
    {
        if (path.find(".log") != std::string::npos)
        {
            return true;
        }

        return false;
    }

    bool IsDatFile(std::string path) NN_NOEXCEPT
    {
        if (path.find(".dat") != std::string::npos)
        {
            return true;
        }

        return false;
    }
}

LoadScene::LoadScene(HashList** ppInHashList) NN_NOEXCEPT
    : m_ppHashList(ppInHashList)
{
    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    m_Label = new glv::Label("Calculating...", glv::Label::Spec(glv::Place::TL, 5, 0, 35));
    *this << m_Label;

    m_Progress = new glv::Label("", glv::Label::Spec(glv::Place::TL, 300, 0, 30));
    *this << m_Progress;
}

nn::Result LoadScene::ReadBinaryStorage(std::unique_ptr<nn::fs::IStorage>& storage, int partitionId, int64_t offset, int64_t totalSize) NN_NOEXCEPT
{
    HashGenerator hashGenerator;
    hashGenerator.Initialize();

    int64_t readSum = 0;
    while (readSum < totalSize)
    {
        int64_t readSize = std::min(StorageReadingBufferSize, totalSize - readSum);
        NN_RESULT_DO(storage.get()->Read(offset + readSum, s_StorageReadingBuffer, readSize));
        hashGenerator.Update(s_StorageReadingBuffer, readSize);

        readSum += readSize;
        char buf[1024];
        nn::util::SNPrintf(buf, 1024, "%lld / %lld", readSum, totalSize);
        m_Progress->setValue(std::string(buf));
    }

    const size_t HashSize = hashGenerator.GetHashSize();
    std::unique_ptr<uint8_t[]> hashBuffer(new uint8_t[HashSize]);
    hashGenerator.GetHash(hashBuffer.get(), HashSize);

    //コンテンツ ID の計算式と合わせる ( sha256 の先頭 128 bit )
    std::string hashValue;
    CalculateContentsId(&hashValue, hashBuffer.get(), HashSize);
    m_ppHashList[partitionId]->AddHash(hashValue.c_str());

    NN_RESULT_SUCCESS;
}

nn::Result LoadScene::ReadFileSystem(HashList *hashList, std::string fileName, nn::fs::DirectoryEntryType type) NN_NOEXCEPT
{
    if (type == nn::fs::DirectoryEntryType_File)
    {
        if (IsSaveData(fileName))
        {
            NN_RESULT_SUCCESS;
        }

        if (IsPrf2Safe(fileName))
        {
            NN_RESULT_SUCCESS;
        }

        if (IsLogFile(fileName))
        {
            NN_RESULT_SUCCESS;
        }

        if (IsDatFile(fileName))
        {
            NN_RESULT_SUCCESS;
        }

        nn::fs::FileHandle handle;
        if (nn::fs::OpenFile(&handle, fileName.c_str(), nn::fs::OpenMode_Read).IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

            int64_t fileSize = 0;
            NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

            int64_t offset = 0;

            HashGenerator hashGenerator;
            hashGenerator.Initialize();

            while (offset < fileSize)
            {
                size_t readSize = std::min(static_cast<int64_t>(StorageReadingBufferSize), fileSize - offset);
                NN_RESULT_DO(nn::fs::ReadFile(handle, offset, s_StorageReadingBuffer, readSize));
                offset += readSize;

                hashGenerator.Update(s_StorageReadingBuffer, readSize);

                //進捗表示
                char buf[1024];
                nn::util::SNPrintf(buf, 1024, "%lld / %lld", offset, fileSize);
                m_Progress->setValue(std::string(buf));
            }

            const size_t HashSize = hashGenerator.GetHashSize();
            std::unique_ptr<uint8_t[]> hashBuffer(new uint8_t[HashSize]);
            hashGenerator.GetHash(hashBuffer.get(), HashSize);

            std::string hashValue;
            CalculateContentsId(&hashValue, hashBuffer.get(), HashSize);
            hashList->AddHash(hashValue.c_str());

            NN_LOG("%s %s\n", hashValue.c_str(), fileName.c_str());

        }
    }

    NN_RESULT_SUCCESS;
};

nn::Result LoadScene::ReadBootPartition1(int partitionId, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::IStorage> storage;
    NN_RESULT_DO(nn::fs::OpenBisPartition(&storage, id));
    // EKS がコピーされる部分の避けるためにオフセットをずらして小さ目に見る(0x450 からが EKS)
    // BCT[0]
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x001000, 0x1000));
    // BCT[1]
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x005000, 0x1000));
    // BCT[2]
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x009000, 0x1000));
    // BCT[3]
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x00D000, 0x1000));


    //TORIAEZU: 修理用は比較しない
    // BCT[4]
    //NN_RESULT_DO(ReadBinaryStorage(storage, hashList, 0x011000, 0x1000));

    // BL normal main
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x100000, 128 * 1024));
    // BL normal sub
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x140000, 128 * 1024));

    NN_RESULT_SUCCESS;
}

nn::Result LoadScene::ReadBootPartition2(int partitionId, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::IStorage> storage;
    NN_RESULT_DO(nn::fs::OpenBisPartition(&storage, id));
    // BL safemode main
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x000000, 128 * 1024));
    // BL safemode sub
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, 0x040000, 128 * 1024));

    //修理用は使用していないため、比較しない
    /*
    // BL repair main
    NN_RESULT_DO(ReadBinaryStorage(storage, hashList, 0x080000, 256 * 1024));
    // BL repair sub
    NN_RESULT_DO(ReadBinaryStorage(storage, hashList, 0x0C0000, 256 * 1024));
    */

    NN_RESULT_SUCCESS;
}

nn::Result LoadScene::ReadPackage2Storage(int partitionId, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    // bootconfig を除外するため offset を 16 Kib ずらす
    int64_t offset = 16 * 1024;

    //Package2 の領域サイズが固定でないため、仮に 1500 Kib のみ固定で読み込む
    int64_t totalSize = 1500 * 1024;

    std::unique_ptr<nn::fs::IStorage> storage;
    NN_RESULT_DO(nn::fs::OpenBisPartition(&storage, id));
    NN_RESULT_DO(ReadBinaryStorage(storage, partitionId, offset, totalSize));

    NN_RESULT_SUCCESS;
}

nn::Result LoadScene::EnumerateDirectoryEntries(HashList *hashList, std::string rootPath) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO(
        nn::fs::OpenDirectory(&directoryHandle, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directoryHandle); };

    int64_t entryCount;
    NN_RESULT_DO(
        nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(entryCount);

    int64_t readEntryCount;
    NN_RESULT_DO(
        nn::fs::ReadDirectory(&readEntryCount, &directoryEntries[0], directoryHandle, directoryEntries.size()));

    directoryEntries.resize(readEntryCount);

    if (rootPath[rootPath.length() - 1] != '/')
    {
        rootPath += "/";
    }

    NN_RESULT_DO(
        ReadFileSystem(hashList, rootPath, nn::fs::DirectoryEntryType_Directory));

    for (auto directoryEntry : directoryEntries)
    {
        switch (directoryEntry.directoryEntryType)
        {
        case nn::fs::DirectoryEntryType_File:
        {
            NN_RESULT_DO(
                ReadFileSystem(hashList, rootPath + directoryEntry.name, nn::fs::DirectoryEntryType_File));
            break;
        }
        case nn::fs::DirectoryEntryType_Directory:
        {
            NN_RESULT_DO(
                EnumerateDirectoryEntries(hashList, rootPath + directoryEntry.name));
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result LoadScene::SelectPartition(int partitionId, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
    case nn::fs::BisPartitionId::BootPartition1Root:
    {
        NN_RESULT_DO(ReadBootPartition1(partitionId, id));
        break;
    }
    case nn::fs::BisPartitionId::BootPartition2Root:
    {
        NN_RESULT_DO(ReadBootPartition2(partitionId, id));
        break;
    }
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
    {
        NN_RESULT_DO(ReadPackage2Storage(partitionId, id));
        break;
    }
    //修理用は使用していないため、比較しない
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
    {
        break;
    }
    default:
    {
        NN_UNEXPECTED_DEFAULT;
    }
    }

    NN_RESULT_SUCCESS;
}

void LoadScene::CalculateHash() NN_NOEXCEPT
{
    for (int i = 0; i < BinaryBisPartitionNum; i++)
    {
        m_ppHashList[i]->SetTitle(BinaryPartitionName[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(SelectPartition(i, BinaryBisId[i]));
        m_ppHashList[i]->Sort();
    }
    for (int i = 0; i < BisPartitionNum; i++)
    {
        int id = i + BinaryBisPartitionNum;
        m_ppHashList[id]->SetTitle(PartitionName[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountBis(MountName, BisId[i]));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        const int bufferSize = 200;
        char buffer[bufferSize];
        snprintf(buffer, bufferSize, "%s:/", MountName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(EnumerateDirectoryEntries(m_ppHashList[id], buffer));
        m_ppHashList[id]->Sort();
    }
}

