﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <functional>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>

class LabelButton : public glv::Button
{
public:
    explicit LabelButton(const std::string& text, int fontSize, int padding) NN_NOEXCEPT
        : Button(glv::Rect(20), true)
    {
        m_pLabel = new glv::Label(text, false);
        this->add(FitButton(m_pLabel, fontSize, padding));
    }

    void setValue(std::string v)
    {
        m_pLabel->setValue(v);
        glv::space_t width = (m_pLabel->width() + 1.f) / 2.f * 2.f;
        glv::space_t height = (m_pLabel->height() + 1.f) / 2.f * 2.f;
        glv::Button::extent(width, height);
    }

protected:
    glv::Label* m_pLabel;

private:
    glv::Label* FitButton(glv::Label* pLabel, float fontSize, glv::space_t padding) NN_NOEXCEPT
    {
        pLabel->size(fontSize);
        pLabel->paddingX(padding);
        pLabel->paddingY(padding);
        pLabel->pos(glv::Place::CL, 0, 0).anchor(glv::Place::CL);
        glv::space_t width = (pLabel->width() + 1.f) / 2.f * 2.f;
        glv::space_t height = (pLabel->height() + 1.f) / 2.f * 2.f;
        glv::Button::extent(width, height);
        return pLabel;
    }
};

class ToggleButton : public LabelButton
{
private:
    bool m_isActive;
    glv::Label m_Label;
    std::string m_ActiveText;
    std::string m_DeactiveText;
public:
    ToggleButton(const std::string& text, const std::string& activeText, const std::string& deactiveText) NN_NOEXCEPT
        : LabelButton(text, 35, 4), m_isActive(true), m_Label(), m_ActiveText(activeText), m_DeactiveText(deactiveText)
    {
        m_Label.setValue(m_ActiveText);
        m_Label.size(30);
        m_Label.anchor(glv::Place::TR);
        *this << m_Label;
    }

    bool isActive()
    {
        return m_isActive;
    }

    virtual bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
    {
        switch (e)
        {
        case glv::Event::MouseUp:
        {
            m_isActive = !m_isActive;

            if (m_isActive)
            {
                m_Label.setValue(m_ActiveText);
            }
            else
            {
                m_Label.setValue(m_DeactiveText);
            }

            break;
        }
        default:
            break;
        }
        return true;
    }
};

class CommandButton : public LabelButton
{
public:
    CommandButton(const std::string& text, std::function<void()> func) NN_NOEXCEPT
        : LabelButton(text, 50, 4), m_func(func)
    {}

    virtual bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
    {
        switch (e)
        {
        case glv::Event::MouseUp:
        {
            if (m_func != nullptr)
            {
                m_func();
            }

            break;
        }
        default:
            break;
        }
        return true;
    }

private:
    std::function<void()> m_func;
};

class VerifyButton : public LabelButton
{
public:
    VerifyButton(const std::string& text, std::function<void(std::string)> func, int fontSize, int padding) NN_NOEXCEPT
        : LabelButton(text, fontSize, padding), m_func(func)
    {}

    virtual bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
    {
        switch (e)
        {
        case glv::Event::MouseUp:
        {
            if (m_func != nullptr)
            {
                m_func(m_pLabel->getValue());
            }

            break;
        }
        default:
            break;
        }
        return true;
    }

    void SetFunction(std::function<void(std::string)>& func)
    {
        m_func = func;
    }

private:
    std::function<void(std::string)> m_func;
};
