﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       振動が付与されたシークバー作成ための API の宣言
 */

#pragma once

#include "hidfw/gfx/gfx_FontSystem.h"
#include "hidfw/layout/layout_SeekBar.h"

#include "VibrationDemo_VibrationItem.h"

namespace VibrationDemo
{
    /**
     * @brief       シークバーを作成するためのクラスです
     */
    class VibrationSeekBar : public VibrationItem
    {
    public:
        static const size_t LimitMaxValue;              //!< MaxValue の上限値
    public:
        VibrationSeekBar() NN_NOEXCEPT;
        virtual ~VibrationSeekBar() NN_NOEXCEPT {}

        bool UpdateControllerEvent() NN_NOEXCEPT;
        bool UpdateTouchEvent() NN_NOEXCEPT;
        void UpdateVibration() NN_NOEXCEPT;

        virtual void Update() NN_NOEXCEPT;
        virtual void PrintText() NN_NOEXCEPT;
        virtual void Draw() NN_NOEXCEPT;

        void SetLimiterSpan(uint64_t value) NN_NOEXCEPT { m_LimiterSpan = value; }

        //----------------------------------------------------------------
        //! @brief  デフォルト値を設定
        //----------------------------------------------------------------
        virtual void SetDefault() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      最大値を設定します
        //! @param[in]  value 最大値
        //----------------------------------------------------------------
        void SetMaxValue(int value) NN_NOEXCEPT { m_MaxValue = value; }

        //----------------------------------------------------------------
        //! @brief      バーの値を設定します
        //! @param[in]  設定値
        //----------------------------------------------------------------
        void SetValue(int value) NN_NOEXCEPT { m_InnerValue = static_cast<double>(value); }

        //----------------------------------------------------------------
        //! @brief      バーの値を加算します
        //! @param[in]  value 加算値
        //----------------------------------------------------------------
        void AddValue(int value) NN_NOEXCEPT { m_InnerValue += static_cast<double>(value); }

        //----------------------------------------------------------------
        //! @brief      バーの値を減算します
        //! @param[in]  value 減算値
        //----------------------------------------------------------------
        void SubValue(int value) NN_NOEXCEPT { m_InnerValue -= static_cast<double>(value); }

        //----------------------------------------------------------------
        //! @brief      バーの値を取得します
        //! @return     値
        //----------------------------------------------------------------
        int GetValue() const NN_NOEXCEPT { return m_Value; }

        //----------------------------------------------------------------
        //! @brief      バーの上側に現在の値を描画します
        //! @param[in]  描画カウント ( showCount / 60 秒)
        //----------------------------------------------------------------
        void ShowValue(uint64_t showCount) NN_NOEXCEPT { m_ValueShowCount = m_FrameCount + showCount; }

        //----------------------------------------------------------------
        //! @brief      値が変化したときに自動的に現在の値を描画するか設定します
        //! @param[in]  有効か否か
        //----------------------------------------------------------------
        void EnableAutoShowValue(bool isEnable) NN_NOEXCEPT { m_IsValueAutoShowEnable = isEnable; }

        //----------------------------------------------------------------
        //! @brief      値の自動描画機能が有効か取得します
        //! @return     値の自動描画が有効な場合 true
        //----------------------------------------------------------------
        bool IsEnableAutoShowValue() const NN_NOEXCEPT { return m_IsValueAutoShowEnable; }

        //----------------------------------------------------------------
        //! @brief      値の描画が有効か
        //! @return     値を描画している最中なら true
        //----------------------------------------------------------------
        bool IsEnableShowValue() const NN_NOEXCEPT { return (m_ValueShowCount > m_FrameCount); }
    protected:
        float               m_BarSize;                  //!< バーの横幅
        float               m_BarPos;                   //!< バーの位置 (最小: 0.f 最大 1.f)

        double              m_InnerValue;               //!< 内部で保持している詳細な現在値
        int                 m_PrevValue;                //!< 1F 前の値
        int                 m_Value;                    //!< 現在値
        int                 m_MaxValue;                 //!< 最大値

        nn::util::Color4u8  m_BaseColor;                //!< メータ部分の下地の色
        nn::util::Color4u8  m_BarColor;                 //!< 操作バーの色

        float               m_MeterSize;                //!< メータサイズ
        nn::util::Float2    m_KnobPos;                  //!< 操作部の表示座標
        nn::util::Float2    m_KnobSize;                 //!< 操作部のサイズ
        nn::util::Float2    m_TextAreaPos;              //!< マージンを含めたテキストの描画座標
        nn::util::Float2    m_TextAreaSize;             //!< マージンを含めたテキストの描画領域
        nn::util::Float2    m_Margin;                   //!< メーターは縦幅が狭くタッチ操作がし難いためマージンを設けます

        bool                m_IsValueAutoShowEnable;    //!< 現在値の自動描画のON/OFF
        uint64_t            m_ValueShowCount;           //!< 値の表示時間
        float               m_TextTransparence;         //!< テキストの透明度

        bool                m_IsTouchItem;              //!< アイテムをタッチしている最中 true
        bool                m_IsLimited;
        int                 m_LimiterSpan;
    };
}
