﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_VibrationSeekBar.h"

namespace VibrationDemo
{
    const size_t VibrationSeekBar::LimitMaxValue = 100000;

    VibrationSeekBar::VibrationSeekBar()
    {
        m_LimiterSpan = 0;
        m_IsLimited = false;
        m_IsTouchItem = false;
        SetDefault();
    }

    void VibrationSeekBar::SetDefault()
    {
        m_Margin = nn::util::MakeFloat2(8.f, 24.f);
        static_cast<BaseItem*>(this)->SetDefault();
        SetSize(320.f, 16.f);

        m_BorderColor       = nn::util::Color4u8( 96,  96,  96, 255);   // ボーダーを濃い灰色に
        m_MainColor         = nn::util::Color4u8( 38, 224,  62, 255);   // メインのゲージ色
        m_SubColor          = nn::util::Color4u8(255,   0,   0, 128);   // ゲージに加算する色
        m_BaseColor         = nn::util::Color4u8( 48,  48,  64, 230);   // ゲージ背面色
        m_BarColor          = nn::util::Color4u8(212, 212, 212, 255);   // 操作バーの色

        m_BarSize           = 16.f;
        m_BarPos            = 0.f;

        m_InnerValue        = 50.0;
        m_Value             = 50;
        m_MaxValue          = 100;

        m_IsValueAutoShowEnable = true;
        m_ValueShowCount = 0;
        m_TextTransparence  = 0.f;
    }

    bool VibrationSeekBar::UpdateControllerEvent() NN_NOEXCEPT
    {
        bool result = false;

        if (
            (m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Hover>()) ||
            (!m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
            )
        {
            if (m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
            {
                m_Option.Set<nns::hidfw::layout::LayoutOption::ThroughChoose>();
            }
            else
            {
                m_Option.Reset<nns::hidfw::layout::LayoutOption::ThroughChoose>();
            }
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                // デジタルキーは小刻みな調整に利用し、最優先で処理します
                if ((*it)->GetButtonSet().IsAnyOn()) { result = true; }
                if ((*it)->IsHold(nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::Right::Mask))
                {
                    if ((*it)->IsHold(nn::hid::NpadButton::Left::Mask))
                    {
                        if ((*it)->IsRepeat(nn::hid::NpadButton::Left::Mask))
                        {
                            SubValue(1);
                        }
                    }
                    else
                    {
                        if ((*it)->IsRepeat(nn::hid::NpadButton::Right::Mask))
                        {
                            AddValue(1);
                        }
                    }
                    if (IsEnableAutoShowValue()) { ShowValue(120); }
                    break;
                }
                else
                {
                    if ((*it)->IsHold(nn::hid::NpadButton::StickLLeft::Mask | nn::hid::NpadButton::StickRLeft::Mask))
                    {
                        SubValue(1);
                        if (IsEnableAutoShowValue()) { ShowValue(120); }
                        break;
                    }
                    else if ((*it)->IsHold(nn::hid::NpadButton::StickLRight::Mask | nn::hid::NpadButton::StickRRight::Mask))
                    {
                        AddValue(1);
                        if (IsEnableAutoShowValue()) { ShowValue(120); }
                        break;
                    }
                }
            }
        }

        return result;
    }

    bool VibrationSeekBar::UpdateTouchEvent() NN_NOEXCEPT
    {
        bool result = false;

        nn::hid::GestureState state[nn::hid::GestureStateCountMax];
        const auto count = gTouch.GetGestureStates(state, nn::hid::GestureStateCountMax, nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(1280.f, 720.f));

        for (int i = 0; i < count; ++i)
        {
            if (state[i].pointCount == 1)
            {   // シングルタッチのみを対象とします
                result = true;

                switch (state[i].GetGestureType())
                {
                case nn::hid::GestureType_Pan:
                    if (m_IsTouchItem)
                    {
                        if (IsEnableAutoShowValue())
                        {
                            ShowValue(120);
                        }
                        const auto lpos = static_cast<float>(state[i].x) - m_Pos.x;
                        m_InnerValue = (lpos / std::max(1.f, m_Size.x)) * static_cast<double>(m_MaxValue);
                    }
                    break;
                case nn::hid::GestureType_Complete:
                    m_IsTouchItem = false;
                    break;
                case nn::hid::GestureType_Cancel:
                    m_IsTouchItem = false;
                    break;
                case nn::hid::GestureType_Touch:
                {
                    const auto length = nn::util::Vector2f(state[i].x - m_KnobPos.x, state[i].y - m_KnobPos.y).Length();
                    if (length < m_Size.y * 4.f)
                    {
                        m_IsTouchItem = true;
                    }
                    break;
                }
                default:
                    break;
                }
            }
            else
            {
                m_IsTouchItem = false;
            }
        }
        return result;
    }

    void VibrationSeekBar::UpdateVibration() NN_NOEXCEPT
    {
        if (m_PrevValue != m_Value)
        {
            SetVibrationBalance(0.f);
            int key = (m_Value == m_MaxValue || m_Value == 0 || (m_LimiterSpan > 0 && m_Value % m_LimiterSpan == 0)) ? nns::hidfw::layout::LayoutState::LimitingValue::Index : nns::hidfw::layout::LayoutState::UpdateValue::Index;

            if (key == nns::hidfw::layout::LayoutState::LimitingValue::Index)
            {
                SetVibrationBalance(((static_cast<float>(m_Value) - static_cast<float>(m_MaxValue / 2)) / static_cast<float>(m_MaxValue)) * 2.f);
            }

            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }

            CallFunc();
            SetVibrationBalance(0.f);
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Hover>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Hover>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Hover::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Selected>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Selected>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Selected::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Canceled>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Canceled>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Canceled::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }
    }

    void VibrationSeekBar::Update()
    {
        m_PrevValue = m_Value;
        ++m_FrameCount;

        if (UpdateControllerEvent() == false)
        {   // コントローラ操作が発生しなかった場合
            if (UpdateTouchEvent() == false)
            {   // タッチ操作が発生しなかった場合
                m_IsLimited = false;
            }
        }

        m_InnerValue = std::max(0.0, std::min(static_cast<double>(m_MaxValue), m_InnerValue));
        m_Value = static_cast<int>(std::round(m_InnerValue));

        if (
            m_IsLimited ||
            (
                m_PrevValue != m_Value &&
                ((m_Value >= m_LimiterSpan && m_PrevValue < m_LimiterSpan) || (m_Value <= m_LimiterSpan && m_PrevValue > m_LimiterSpan))
            )
            )
        {
            m_IsLimited = true;
            m_InnerValue = static_cast<double>(m_LimiterSpan);
            m_Value = m_LimiterSpan;
        }

        m_MeterSize = (m_Size.x - m_Size.y * 2.f) * (static_cast<float>(m_Value) / static_cast<float>(std::max(1, m_MaxValue))) + m_Size.y;
        m_KnobSize = nn::util::MakeFloat2(m_Size.y * 2.f, m_Size.y * 1.5f);
        m_KnobPos = nn::util::MakeFloat2(m_Pos.x + m_MeterSize - m_KnobSize.x / 2.f, m_Pos.y - (m_KnobSize.y - m_Size.y) / 2.f);

        if (m_State.Test<nns::hidfw::layout::LayoutState::Selected>() && !m_State.Test<nns::hidfw::layout::LayoutState::OnFocus>() && !m_State.Test<nns::hidfw::layout::LayoutState::Hover>())
        {
            m_State.Reset<nns::hidfw::layout::LayoutState::Selected>();
            m_Option.Reset<nns::hidfw::layout::LayoutOption::ThroughChoose>();
        }

        if (IsEnableShowValue())
        {
            m_TextTransparence = std::min(static_cast<float>(m_ValueShowCount - m_FrameCount) / 60.f, 0.8f);
        }
        else
        {
            m_TextTransparence = 0.f;
        }

        UpdateVibration();
    }

    void VibrationSeekBar::PrintText()
    {
        if (IsEnableShowValue())
        {
            SetText("%d / %d", m_Value, m_MaxValue);
            m_TextTransparence = m_TextTransparence - std::floorf(m_TextTransparence);
            gTextWriter.SetTextColor(nn::util::Color4u8(m_TextColor.GetR(), m_TextColor.GetG(), m_TextColor.GetB(), static_cast<uint8_t>(m_TextColor.GetA() * m_TextTransparence)));
            gTextWriter.SetScale(m_Scale.x, m_Scale.y);

            m_TextAreaSize = nn::util::MakeFloat2(
                gTextWriter.CalculateStringWidth(m_Text.c_str()) + 32.f,
                gTextWriter.CalculateStringHeight(m_Text.c_str()) + 32.f);

            m_TextAreaPos = nn::util::MakeFloat2(
                m_Pos.x + (m_BarPos * m_Size.x) - (m_TextAreaSize.x / 2.f) + m_MeterSize,
                (m_Pos.y - m_TextAreaSize.y > m_Margin.y) ?
                m_Pos.y - m_TextAreaSize.y - m_Margin.y - 16.f :
                m_Pos.y + m_Size.y + m_Margin.y - 16.f
            );

            const nn::util::Float2 textPos = nn::util::MakeFloat2(
                m_TextAreaPos.x + 16.f,
                m_TextAreaPos.y + 16.f);

            gTextWriter.SetCursor(textPos.x, textPos.y);
            gTextWriter.Print("%s", m_Text.c_str());
        }
        m_OldState = m_State;
    }

    void VibrationSeekBar::Draw()
    {
        const auto effectColor = m_State.Test<nns::hidfw::layout::LayoutState::Selected>() ? Color::Yellow : Color::Cyan;
        const auto transparenceEffectColor = nn::util::Color4u8(effectColor.GetR(), effectColor.GetG(), effectColor.GetB(), 0);

        const auto borderColor = m_BorderColor;
        const auto transparenceBorderColor = nn::util::Color4u8(borderColor.GetR(), borderColor.GetG(), borderColor.GetB(), 0);

        const auto backgroundColor = nn::util::Color4u8(16, 16, 16, 255);
        const auto endBackgroundColor = nn::util::Color4u8(32, 32, 48, 255);

        const auto mainColor = m_MainColor;
        const auto shadowColor = nn::util::Color4u8(mainColor.GetR() / 3, mainColor.GetG() / 3, mainColor.GetB() / 3, mainColor.GetA());


        if (m_State.Test<nns::hidfw::layout::LayoutState::Selected>() || m_State.Test<nns::hidfw::layout::LayoutState::Hover>())
        {   // 選択状態
            auto pos = nn::util::MakeFloat2(m_Pos.x - 7.f, m_Pos.y - 7.f);
            auto size = nn::util::MakeFloat2(m_Size.x + 14.f, m_Size.y + 14.f);
            gDrawer.SetColor(effectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, 2.f);

            pos = nn::util::MakeFloat2(m_Pos.x - 10.f, m_Pos.y - 10.f);
            size = nn::util::MakeFloat2(m_Size.x + 20.f, m_Size.y + 20.f);
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, transparenceEffectColor, effectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, -1.f);

            pos = nn::util::MakeFloat2(m_Pos.x - 6.f, m_Pos.y - 6.f);
            size = nn::util::MakeFloat2(m_Size.x + 12.f, m_Size.y + 12.f);
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, transparenceEffectColor, effectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, 1.f);
        }

        // 下地
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, backgroundColor, endBackgroundColor);
        gDrawer.Draw2DRoundedRect(m_Pos, m_Size, 1.f, 64);

        // メータ
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, mainColor, shadowColor);
        gDrawer.Draw2DRoundedRect(m_Pos, nn::util::MakeFloat2(m_MeterSize, m_Size.y), 1.f, 64);

        // 枠
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, borderColor, transparenceBorderColor);
        gDrawer.Draw2DRoundedFrame(m_Pos, m_Size, 1.f, 64, 1.f);
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_In, borderColor, transparenceBorderColor);
        gDrawer.Draw2DRoundedFrame(m_Pos, m_Size, 1.f, 64, -1.f);

        // 操作部
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, borderColor, transparenceBorderColor);
        gDrawer.Draw2DRoundedFrame(m_KnobPos, m_KnobSize, 0.5f, 16, 1.f);
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, nn::util::Color4u8(172, 172, 172, 255), borderColor);
        gDrawer.Draw2DRoundedRect(m_KnobPos, m_KnobSize, 0.5f, 16);
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, borderColor, transparenceBorderColor);
        gDrawer.Draw2DRoundedFrame(m_KnobPos, m_KnobSize, 0.5f, 16, -1.f);

        if (IsEnableShowValue())
        {
            // 値の描画が必要な場合バーより後に描画処理を実行します
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down,
                nn::util::Color4u8(255, 255, 255, static_cast<uint8_t>(m_TextColor.GetA() * m_TextTransparence)),
                nn::util::Color4u8(255, 255, 255, 0));
            gDrawer.Draw2DRoundedFrame(m_TextAreaPos, m_TextAreaSize, 0.5f, 48, 3.f);
            gDrawer.SetColor(nn::util::Color4u8(255, 255, 255, static_cast<uint8_t>(m_TextColor.GetA() * m_TextTransparence)));
            gDrawer.Draw2DRoundedRect(m_TextAreaPos, m_TextAreaSize, 0.5f, 48);
        }
    }
}
