﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_VibrationManager.h"

namespace VibrationDemo
{
    NN_OS_ALIGNAS_THREAD_STACK char VibrationManager::ThreadStack[VibrationThreadStackSize];
    nn::os::ThreadType              VibrationManager::VibrationThread;
    nn::os::TimerEventType          VibrationManager::VibrationTimerEvent;
    BnvibFileHeap<VibrationManager::VibrationMemoryHeapSize>  VibrationManager::VibrationHeap;
    std::map<std::string, BnvibFile> VibrationManager::VibrationFileList;

    std::map<VibrationKeyType, VibrationValues<VibrationManager::VibrationValueCountMax, VibrationManager::HandleCountMax>> VibrationManager::CurrentVibrationValues;
    int64_t VibrationManager::SamplingNumber = 0;

    VibrationManager& VibrationManager::GetInstance() NN_NOEXCEPT
    {
        static VibrationManager instance;
        return instance;
    }

    void VibrationManager::UpdateVibrationNode(void *arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitTimerEvent(&VibrationTimerEvent);
            GetInstance().UpdateNpadStyle();
            nn::hid::VibrationNode::Update();

            for (const auto& dev : GetInstance().GetVibrationDevices())
            {
                NN_ASSERT_LESS_EQUAL(dev.second->handleCount, 2);
                const auto& key = dev.first;

                if (CurrentVibrationValues.find(key) == CurrentVibrationValues.end())
                {
                    CurrentVibrationValues.emplace(key, VibrationValues<VibrationManager::VibrationValueCountMax, VibrationManager::HandleCountMax>());
                }
                int handleNumber = 0;
                for (const auto& handle : dev.second->handles)
                {
                    CurrentVibrationValues.at(key).samplingNumber[SamplingNumber % VibrationValueCountMax] = SamplingNumber;
                    auto& value = CurrentVibrationValues.at(key).vibrationValue[SamplingNumber % VibrationValueCountMax][handleNumber++];
                    nn::hid::GetActualVibrationValue(&value, handle);
                }
            }
            ++SamplingNumber;
        }
    }

    const std::map<std::string, BnvibFile>& VibrationManager::GetBnvibFileList() NN_NOEXCEPT
    {
        return VibrationFileList;
    }

    void VibrationManager::UpdateNpadStyle() NN_NOEXCEPT
    {
        static nn::hid::NpadStyleSet prevStyleSet[NpadCountMax];
        for (size_t i = 0; i < m_IsEnableVibrationNpadIds.size(); ++i)
        {   // 全てのIDに対してスタイルの状態が変化しているか検査します
            auto npadId = m_IsEnableVibrationNpadIds.at(i);
            auto styleSet = nn::hid::GetNpadStyleSet(npadId);
            if (
                (prevStyleSet[i].IsAnyOn() || styleSet.IsAnyOn()) &&
                ((prevStyleSet[i] & styleSet).IsAllOff())
                )
            {   // スタイルが変化している場合
                nn::hid::NpadStyleSet nextStyle;
                for (int styleIndex = 0; static_cast<size_t>(styleIndex) < NpadStyleCountMax; ++styleIndex)
                {   // 全スタイルを検査し、変化後のスタイルと振動ハンドルを対応付ける
                    nextStyle.Reset();
                    nextStyle.Set(styleIndex);
                    auto key = VibrationKeyType::Make(npadId, nextStyle);
                    if (m_VibrationMixerConnections.find(key) != m_VibrationMixerConnections.end())
                    {   // スタイルとIDが登録されたものである場合
                        if ((nextStyle & styleSet).IsAnyOn())
                        {   // 変化後のスタイルに対してハンドルを対応付ける
                            for (size_t j = 0; j < m_VibrationMixerConnections.at(key).size(); ++j)
                            {
                                m_VibrationMixerConnections.at(key).at(j)->Connect(&m_VibrationMixer[i][j], m_VibrationTargets.at(key).at(j));
                                m_VibrationTargets.at(key).at(j)->SetVibrationDeviceHandle(m_pVibrationDevice.at(key)->handles[j]);
                            }
                        }
                        else
                        {   // 関係の無いスタイルに対して対応付けられているハンドルを解除する
                            for (size_t j = 0; j < m_VibrationMixerConnections.at(key).size(); ++j)
                            {
                                m_VibrationMixerConnections.at(key).at(j)->Disconnect();
                                m_VibrationTargets.at(key).at(j)->UnsetVibrationDeviceHandle();
                            }
                        }
                    }
                }
                prevStyleSet[i] = styleSet;
            }
        }
    }

    void VibrationManager::Initialize(const nn::hid::NpadIdType* npadIds, const int npadIdCount, const nn::hid::NpadStyleSet npadStyleSet) NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            return;
        }
        m_IsInitialized = true;
        NN_ASSERT_NOT_NULL(npadIds);
        NN_ASSERT_RANGE(npadIdCount, 0, 10);
        NN_ASSERT(npadStyleSet.CountPopulation() > 0);

        nn::hid::NpadStyleSet npadStyles[NpadStyleCountMax];

        CurrentVibrationValues.clear();

        m_NpadIndices.clear();
        m_pVibrationDevice.clear();
        m_NpadIdCount = npadIdCount;
        m_NpadStyleCount = npadStyleSet.CountPopulation();

        int counter = 0;
        for (auto& style : npadStyles)
        {
            style.Reset();
            for (; counter < style.GetCount(); ++counter)
            {
                if (npadStyleSet.Test(counter))
                {
                    style.Set(counter);
                    ++counter;
                    break;
                }
            }
        }

        for (auto& mixers : m_VibrationMixer)
        {
            for (auto& mixer : mixers)
            {
                mixer.SetMixMode(nn::hid::VibrationMixMode_MaxAmplitudePerSubband);
            }
        }

        for (int i = 0; i < m_NpadIdCount; ++i)
        {
            auto npadId = npadIds[i];
            // フレームワークの振動機能と競合するため無効化します
            gController.GetController(npadId)->GetVibration().Disable();

            // デバイスデータの実態を作成します
            if (m_pVibrationDeviceData[i] != nullptr)
            {
                delete[] m_pVibrationDeviceData[i];
                m_pVibrationDeviceData[i] = nullptr;
            }
            m_pVibrationDeviceData[i] = new VibartionDevice[NN_ARRAY_SIZE(npadStyles)];

            // npadId のインデックスをリストに追加します
            m_NpadIndices.insert(std::make_pair(npadId, i));

            // 各スタイル分の振動子ハンドルを取得します
            for (int styleIndex = 0; styleIndex < m_NpadStyleCount; ++styleIndex)
            {
                auto style = npadStyles[styleIndex];
                auto key = VibrationKeyType::Make(npadId, style);

                m_pVibrationDevice.emplace(std::make_pair(key, &m_pVibrationDeviceData[i][styleIndex]));
                NN_ASSERT(m_pVibrationDevice.find(key) != m_pVibrationDevice.end());

                auto device = m_pVibrationDevice.at(key);

                device->handleCount = nn::hid::GetVibrationDeviceHandles(device->handles, HandleCountMax, npadId, style);

                auto target = m_VibrationTargets.emplace(key, NULL);
                if (target.second)
                {
                    m_VibrationMixerConnections.emplace(key, NULL);
                    target.first->second.clear();
                    for (int handleIndex = 0; handleIndex < device->handleCount; ++handleIndex)
                    {
                        target.first->second.emplace_back(new nn::hid::VibrationTarget(device->handles[handleIndex]));
                        target.first->second.back()->SetMixMode(nn::hid::VibrationMixMode_MaxAmplitudePerSubband);
                        m_VibrationMixerConnections.at(key).emplace_back(new nn::hid::VibrationNodeConnection());
                    }
                }
            }
        }

        EnableVibration();
        // タイマーイベントを初期化する
        nn::os::InitializeTimerEvent(&VibrationTimerEvent, nn::os::EventClearMode_AutoClear);
        // スレッドを生成する
        nn::Result result = nn::os::CreateThread(&VibrationThread, UpdateVibrationNode, NULL, ThreadStack, VibrationThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    }

    size_t VibrationManager::LoadBnvibForMemory() NN_NOEXCEPT
    {
        size_t resultFileCount = 0;
        char* cache = new char[1024 * 1024 * 16];
        NN_UNUSED(cache);

        nn::Result result;
        result = nn::fs::MountRom("BNVB", cache, 1024 * 1024 * 8);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        {
            nn::fs::DirectoryHandle handle;

            result = nn::fs::OpenDirectory(&handle, "BNVB:/bnvib/", nn::fs::OpenDirectoryMode_File);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ファイルが存在しません。
                // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
                NN_ASSERT(false, "File not found.\n");
                return 0;
            }

            if (result.IsSuccess())
            {
                int64_t fileCount = 0;
                nn::fs::DirectoryEntry fileEntry[256];

                result = nn::fs::ReadDirectory(&fileCount, fileEntry, handle, NN_ARRAY_SIZE(fileEntry));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                resultFileCount = fileCount;
                for (int64_t i = 0; i < fileCount; ++i)
                {
                    nn::fs::FileHandle fileHandle;
                    if (VibrationFileList.find(fileEntry[i].name) == VibrationFileList.end())
                    {
                        result = nn::fs::OpenFile(&fileHandle, (std::string("BNVB:/bnvib/") + fileEntry[i].name).c_str(), nn::fs::OpenMode_Read);
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                        {
                            std::string fileName = fileEntry[i].name;
                            int64_t fileSize = 0;
                            nn::fs::GetFileSize(&fileSize, fileHandle);
                            uint8_t* heap = (uint8_t*)VibrationHeap.Allocate(fileSize);
                            nn::fs::ReadFile(fileHandle, 0, heap, fileSize);

                            BnvibFile fileData;
                            fileData.address = heap;
                            fileData.fileSize = fileSize;
                            VibrationFileList.emplace(std::make_pair(fileName, fileData));
                        }
                        nn::fs::CloseFile(fileHandle);
                    }
                }
                nn::fs::CloseDirectory(handle);
            }
        }
        nn::fs::Unmount("BNVB");
        return resultFileCount;
    }

    bool VibrationManager::GetBnvibFile(const char* fileName, void** address, size_t* outFileSize) NN_NOEXCEPT
    {
        auto file = VibrationFileList.find(fileName);
        if (file != VibrationFileList.end())
        {
            *address = file->second.address;
            *outFileSize = file->second.fileSize;
            return true;
        }
        return false;
    }

    void VibrationManager::StartUpdateThread() NN_NOEXCEPT
    {
        // スレッドの実行を開始する
        nn::os::StartThread(&VibrationThread);
        // タイマーイベントは周期タイマーイベントとして開始する
        const nn::TimeSpan interval = nn::hid::VibrationNode::DefaultVibrationSampleInterval;
        nn::os::StartPeriodicTimerEvent(&VibrationTimerEvent, interval, interval);
    }

    void VibrationManager::StopUpdateThread() NN_NOEXCEPT
    {
        if (VibrationTimerEvent._state == VibrationTimerEvent.State_Initialized)
        {
            nn::os::StopTimerEvent(&VibrationTimerEvent);
        }
    }

    void VibrationManager::StopDefaultVibrationNodes() NN_NOEXCEPT
    {
        for (auto& writer : m_DefaultVibrationWriter)
        {
            writer.Write(nn::hid::VibrationValue::Make());
        }
        for (auto& player : m_DefaultVibrationPlayer)
        {
            if (player.IsLoaded() && player.IsPlaying())
            {
                player.Stop();
            }
        }
    }

    void VibrationManager::DisconnectAll() NN_NOEXCEPT
    {
        for (auto connectionItr : m_SourceConnections)
        {
            if (connectionItr != nullptr)
            {
                connectionItr->Disconnect();
            }
        }
    }

    nn::hid::VibrationNode* VibrationManager::GetVibrationMixer(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType type) NN_NOEXCEPT
    {
        // 有効でないコントローラが指定された場合アサートする
        NN_ASSERT_EQUAL(m_NpadIndices.count(id), (size_t)1);

        auto itr = m_NpadIndices.find(id);
        if (itr != m_NpadIndices.end())
        {
            auto index = itr->second;

            switch (type)
            {
            case nn::hid::NpadJoyDeviceType_Left:
                return &m_VibrationMixer[index][0];
                break;
            case nn::hid::NpadJoyDeviceType_Right:
                return &m_VibrationMixer[index][1];
                break;
            default:NN_UNEXPECTED_DEFAULT;
            }
        }
        return nullptr;
    }

    bool VibrationManager::IsEnableVibrationNpadId(nn::hid::NpadIdType id) NN_NOEXCEPT
    {
        auto itr = std::find(m_IsEnableVibrationNpadIds.begin(), m_IsEnableVibrationNpadIds.end(), id);
        return (itr != m_IsEnableVibrationNpadIds.end());
    }

    int VibrationManager::GetEnableVibrationNpadIds(nn::hid::NpadIdType* pOutIds, const int count) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutIds);
        int enableCount = 0;

        for (auto id : m_IsEnableVibrationNpadIds)
        {
            if (enableCount >= count)
            {
                break;
            }
            pOutIds[enableCount++] = id;
        }
        return enableCount;
    }

    void VibrationManager::EnableVibration(nn::hid::NpadIdType* pIds, const int count) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pIds);
        for (int i = 0; i < count; ++i)
        {
            auto itr = std::find(m_IsEnableVibrationNpadIds.begin(), m_IsEnableVibrationNpadIds.end(), pIds[i]);
            if (itr == m_IsEnableVibrationNpadIds.end())
            {
                m_IsEnableVibrationNpadIds.push_back(pIds[i]);
            }
        }
    }

    void VibrationManager::EnableVibration() NN_NOEXCEPT
    {
        for (auto itr : m_NpadIndices)
        {
            EnableVibration((nn::hid::NpadIdType*)(&itr.first), 1);
        }
        for (auto& connection : m_VibrationMixerConnections)
        {
            for (auto& c : connection.second)
            {
                if (c->IsConnected())
                {
                    c->SetModulation(nn::hid::VibrationModulation::Make());
                }
            }
        }
    }

    void VibrationManager::DisableVibration(nn::hid::NpadIdType* pIds, const int count) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pIds);
        for (int i = 0; i < count; ++i)
        {
            auto itr = std::find(m_IsEnableVibrationNpadIds.begin(), m_IsEnableVibrationNpadIds.end(), pIds[i]);
            while (itr != m_IsEnableVibrationNpadIds.end())
            {
                m_IsEnableVibrationNpadIds.erase(itr);
                itr = std::find(m_IsEnableVibrationNpadIds.begin(), m_IsEnableVibrationNpadIds.end(), pIds[i]);
            }
        }
    }

    void VibrationManager::DisableVibration() NN_NOEXCEPT
    {
        m_IsEnableVibrationNpadIds.clear();

        for (auto& connection : m_VibrationMixerConnections)
        {
            for (auto& c : connection.second)
            {
                if (c->IsConnected())
                {
                    c->SetModulation(nn::hid::VibrationModulation::Make(0.f, 0.f, 0.f, 0.f));
                }
            }
        }
    }

    nn::hid::VibrationWriter& VibrationManager::GetVibrationWriter(nn::hid::NpadIdType id) NN_NOEXCEPT
    {
        auto style = gController.GetController(id)->GetStyleSet();
        auto key = VibrationKeyType::Make(id, style);
        NN_UNUSED(key);
        // 有効でないコントローラが指定された場合アサートする
        NN_ASSERT_EQUAL(m_NpadIndices.count(id), (size_t)1);
        NN_ASSERT_EQUAL(m_pVibrationDevice.count(key), (size_t)1);

        auto index = m_NpadIndices.at(id);
        auto& writer = m_DefaultVibrationWriter[index];

        return writer;
    }

    int VibrationManager::ConnectVibrationNode(nn::hid::VibrationNode* pNode, nn::hid::VibrationNodeConnection* pConnections, nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType type) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pNode);
        NN_ASSERT_NOT_NULL(pConnections);

        auto supportedStyleSet = nn::hid::GetSupportedNpadStyleSet();
        nn::hid::NpadStyleSet style;
        int handleCount = 0;

        int i = 0;
        int j = 0;

        for (i = 0; i < supportedStyleSet.GetCount() && j < supportedStyleSet.CountPopulation(); ++i)
        {
            if (supportedStyleSet.Test(i))
            {   // 有効なNpadIdが選択されている場合
                style.Reset();
                style.Set(i);
                auto key = VibrationKeyType::Make(id, style);
                auto deviceItr = m_pVibrationDevice.find(key);
                if (deviceItr != m_pVibrationDevice.end())
                {   // 振動デバイスが登録されている場合
                    auto index = m_NpadIndices.at(id);
                    // 左コンのハンドルインデックスが0、右コンが1を使用しています
                    auto handleIndex = (type == nn::hid::NpadJoyDeviceType_Left || style.Test<nn::hid::NpadStyleJoyLeft>()) ? 0 : 1;

                    if (pConnections->IsConnected())
                    {   // 既に接続が確立されている場合
                        if (pConnections->GetDestination()->IsConnectedTo(pNode))
                        {   // 別の振動子と接続されている場合は切断
                            pConnections->Disconnect();
                            pConnections->Connect(pNode, &m_VibrationMixer[index][handleIndex]);
                        }
                    }
                    else
                    {   // 接続が確立されていない場合はそのまま接続
                        pConnections->Connect(pNode, &m_VibrationMixer[index][handleIndex]);
                    }
                    ++handleCount;
                }
                ++j;
            }
        }

        return handleCount;
    }

    int VibrationManager::ConnectVibrationNode(nn::hid::VibrationNode* pNode, nn::hid::VibrationNodeConnection* pConnections, nn::hid::NpadIdType id) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pNode);
        NN_ASSERT_NOT_NULL(pConnections);

        int connectCount = 0;

        if (m_NpadIndices.count(id) == 1)
        {
            if (pConnections->IsConnected())
            {
                if (pConnections->GetDestination()->IsConnectedTo(pNode))
                {   // 別の振動子と接続されている場合は切断
                    pConnections->Disconnect();
                }
                else
                {   // 既に接続済み
                    return 0;
                }
            }
            for (int i = 0; i < HandleCountMax; ++i)
            {
                pConnections[i].Connect(pNode, &m_VibrationMixer[m_NpadIndices.at(id)][i]);
                ++connectCount;
            }
        }
        return connectCount;
    }

    void VibrationManager::AddSourceConnections(nn::hid::VibrationNodeConnection* pConnection) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pConnection);
        m_SourceConnections.push_back(pConnection);
    }

    void VibrationManager::RemoveSourceConnections(nn::hid::VibrationNodeConnection* pConnection) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pConnection);
        m_SourceConnections.erase(std::find(m_SourceConnections.begin(), m_SourceConnections.end(), pConnection));
    }

    nn::hid::VibrationValue VibrationManager::GetActualVibrationValue(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT
    {
        nn::hid::VibrationValue value = nn::hid::VibrationValue::Make();

        bool isUseLeft = (deviceType != nn::hid::NpadJoyDeviceType_Right) && !nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyRight>();

        const int handleIndex = isUseLeft ? 0 : 1;

        const auto& itr = m_NpadIndices.find(id);
        if (itr != m_NpadIndices.end())
        {
            const auto npadIndex = itr->second;
            if (m_pVibrationDeviceData[npadIndex]->handleCount > handleIndex)
            {
                nn::hid::GetActualVibrationValue(&value, m_pVibrationDeviceData[npadIndex]->handles[handleIndex]);
            }
        }
        return value;
    }

    int64_t VibrationManager::GetActualVibrationValues(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType deviceType, nn::hid::VibrationValue* pValues, int64_t maxCount, int64_t samplingNumber, int64_t* outLastSamplingNumber) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pValues);

        bool isUseLeft = (deviceType != nn::hid::NpadJoyDeviceType_Right) && !nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyRight>();

        const auto handleIndex = isUseLeft ? 0 : 1;
        const auto count = std::max((int64_t) 0, maxCount);

        const auto key = VibrationKeyType::Make(id, nn::hid::GetNpadStyleSet(id));

        if (CurrentVibrationValues.find(key) == CurrentVibrationValues.end())
        {
            return 0;
        }

        const auto& values = CurrentVibrationValues.at(key);

        // 最新のサンプリングナンバーを検索します
        int64_t num = 0;
        int64_t index = 0;

        for (int i = 0; i < VibrationValueCountMax; ++i)
        {
            if (values.samplingNumber[i] > num)
            {
                num = values.samplingNumber[i];
                index = i;
            }
        }

        int64_t curIndex = 0;
        for (int i = index; i > index - VibrationValueCountMax; --i)
        {
            int targetIndex = (i + VibrationValueCountMax) % VibrationValueCountMax;
            if (curIndex >= count || values.samplingNumber[targetIndex] <= samplingNumber)
            {
                break;
            }
            pValues[curIndex++] = values.vibrationValue[targetIndex][handleIndex];
        }

        if (outLastSamplingNumber != nullptr)
        {
            *outLastSamplingNumber = num;
        }
        return curIndex;
    }
}
