﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo.h"

namespace VibrationDemo
{
    void Initialize() NN_NOEXCEPT
    {
        gInput.Initialize();                        //!< 入力管理機能の初期化
        gGraphics.Initialize();                     //!< グラフィックス描画機能の初期化
        gAudioManager.Initialize();                 //!< オーディオ機能の初期化
        gFont.Initialize();                         //!< フォント描画用機能の初期化
        gApplicationState.Initialize();             //!< アプリ情報の初期化
        gControllerSequenceManager.Initialize();    //!< コンサポ管理機能の初期化

        //------------------------------
        // コントローラの初期設定
        //------------------------------
        const nn::hid::NpadStyleSet styleSet =
        {
            nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleHandheld::Mask |
            nn::hid::NpadStyleJoyDual::Mask
        };
        const nn::hid::NpadIdType npadIds[] =
        {
            nn::hid::NpadId::No1,
            nn::hid::NpadId::No2,                   //!< コントローラが追加接続された場合の検出用
            nn::hid::NpadId::Handheld
        };
        nn::hid::SetSupportedNpadStyleSet(styleSet);
        nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));
        // 設定後更新を行う
        gInput.Update();

        //------------------------------
        // 振動管理機能の初期設定
        //------------------------------
        for (auto& controller : gController.GetConnectedControllerList())
        {   // フレームワークの振動機能を無効化します
            controller->GetVibration().Disable();
        }
        gVibrationManager.Initialize(npadIds, NN_ARRAY_SIZE(npadIds), nn::hid::GetSupportedNpadStyleSet());

        //------------------------------
        // ファイル読み込み
        //------------------------------
        gVibrationManager.LoadBnvibForMemory();
        gAudioManager.LoadSoundFilesFromResource();

        //------------------------------
        // シーンの生成
        //------------------------------
        CreateScene();
        // 最初に呼び出されるシーンを設定
        gApplicationState.SetScene(Scene::Ui);

        //------------------------------
        // コンサポ管理機能の設定
        //------------------------------
        ControllerSequenceManagerArg arg;
        arg.SetDefault();
        arg.isSingleMode = true;
        gControllerSequenceManager.SetArgument(arg);
        gControllerSequenceManager.SetSupportedNpadIds(npadIds, NN_ARRAY_SIZE(npadIds));

        //------------------------------
        // 各種機能の動作を開始する
        //------------------------------
        gVibrationManager.StartUpdateThread();
        gAudioManager.StartUpdateThread();
        gControllerSequenceManager.StartControllerCheckThread();

        // 画面の背景色を黒色に設定
        gDrawer.SetClearColor(nn::util::Color4u8::Black());

        // アンチエイリアスを有効化
        gDrawer.SetAntiAliasingType(nns::hidfw::gfx::GraphicsDrawer::AntiAliasingType_Pseudo);
    }

    void Finalize() NN_NOEXCEPT
    {
        gAudioManager.Finalize();
        gGraphics.Finalize();
        gInput.Finalize();
    }

    extern "C" void nnMain() NN_NOEXCEPT
    {
        Initialize();

        do
        {
            NN_ASSERT_NOT_NULL(gApplicationState.GetScene());
            if (gApplicationState.GetScene()->DoMainLoop() == false)
            {
                gApplicationState.Exit();
            }
        } while (gApplicationState.IsExit() == false);

        Finalize();
    }
}
