﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * 振動の処理
 */

#include "Audio.h"
#include "Button.h"
#include "File.h"
#include "Vibration.h"

namespace VibrationCollection
{
    VibrationTarget                             g_VibrationTarget;
    nns::hid::SpeedChangeableVibrationPlayer    g_VibrationPlayer;

    NN_OS_ALIGNAS_THREAD_STACK char             VibrationUpdateThread::ThreadStack[StackSize];
    nn::os::ThreadType                          VibrationUpdateThread::VibrationThread;
    nn::os::TimerEventType                      VibrationUpdateThread::VibrationTimerEvent;

    void VibrationUpdateThread::VibrationNodeThreadFunction(void *arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitTimerEvent(&VibrationTimerEvent);
            nn::hid::VibrationNode::Update();
        }
    }

    void VibrationUpdateThread::Initialize() NN_NOEXCEPT
    {
        // タイマーイベントを初期化する
        nn::os::InitializeTimerEvent(&VibrationTimerEvent, nn::os::EventClearMode_AutoClear);
        // スレッドを生成する
        NN_ASSERT(nn::os::CreateThread(&VibrationThread, VibrationNodeThreadFunction, NULL, ThreadStack, StackSize, nn::os::DefaultThreadPriority).IsSuccess(), "Cannot create thread.");
    }

    void VibrationUpdateThread::Finalize() NN_NOEXCEPT
    {
        nn::os::StopTimerEvent(&VibrationTimerEvent);
    }

    void VibrationUpdateThread::Start() NN_NOEXCEPT
    {
        // スレッドの実行を開始する
        nn::os::StartThread(&VibrationThread);
        // タイマーイベントは周期タイマーイベントとして開始する
        const nn::TimeSpan interval = nn::hid::VibrationNode::DefaultVibrationSampleInterval;
        nn::os::StartPeriodicTimerEvent(&VibrationTimerEvent, interval, interval);
    }

    void StartSwingVibration() NN_NOEXCEPT
    {
        for (auto itr : gController.GetConnectedControllerList())
        {
            itr->StartSixAxisSensor();

            nn::hid::SixAxisSensorState state[2];
            auto count = itr->GetSixAxisSensorState(NN_ARRAY_SIZE(state), state);
            if (count >= 1)
            {
                auto secondSpeed = std::sqrt(std::pow(state[0].acceleration.x, 2) + std::pow(state[0].acceleration.y, 2) + std::pow(state[0].acceleration.z, 2));

                if (secondSpeed > 1.75f && !g_VibrationPlayer.IsPlaying())
                {
                    PlayVibration(nullptr, nullptr);
                }
            }
        }
    }

    void PlayVibration(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        NN_ASSERT_EQUAL(g_DialogSeekBar[0].GetListSize(), SettingButtonType_Num / 2);
        NN_ASSERT_EQUAL(g_DialogSeekBar[1].GetListSize(), SettingButtonType_Num / 2);

        bool isLoopPlay = g_VibrationPlayer.IsPlaying() && g_VibrationPlayer.IsLoop();

        g_VibrationPlayer.Stop();
        g_VibrationPlayer.Unload();

        if (gFileManager.GetBnvibFileCount() > 0)
        {
            NN_ASSERT_RANGE(g_SelectedBnvibButtonInedx, 0, gFileManager.GetBnvibFileCount());

            g_VibrationPlayer.Load(
                gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileData,
                gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileSize);

            auto loopSetting = gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop;

            g_VibrationPlayer.SetLoop(loopSetting.isLoop != 0);
            g_VibrationPlayer.SetLoopStartPosition(loopSetting.loopStartPosition);
            g_VibrationPlayer.SetLoopEndPosition(loopSetting.loopEndPosition);
            g_VibrationPlayer.SetLoopInterval(loopSetting.loopInterval);

            g_VibrationPlayer.ResetOffset();
        }

        if (!isLoopPlay)
        {
            g_VibrationPlayer.Play();
            if (
                gFileManager.GetAudioFileCount() > 0 &&
                g_SelectedAudioButtonIndex < gFileManager.GetAudioFileCount()
                )
            {
                gAudio.PlayWav(g_SelectedAudioButtonIndex);
            }
        }
    }

    void ExportVibrationValue(
        uint8_t* outBuffer,
        size_t bufferSize,
        size_t* pOutSize
    ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outBuffer);
        NN_ASSERT_NOT_NULL(pOutSize);
        NN_ASSERT_RANGE(g_SelectedBnvibButtonInedx, 0, gFileManager.GetBnvibFileCount());
        // 速度が0の場合は出力しない
        NN_ASSERT_GREATER(g_VibrationPlayer.GetPlaySpeed(), 0.f);

        if (!g_VibrationPlayer.IsLoaded())
        {
            g_VibrationPlayer.Load(
                gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileData,
                gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileSize);
        }

        // 出力サイズ (Meta領域 + データ領域)
        size_t& outSize = *pOutSize;
        // ベースとなるBnvib
        auto& baseBnvibFileData = gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx);
        // オフセットのデータサイズを計算します
        uint32_t offsetSize = (g_VibrationPlayer.GetOffset().GetMilliSeconds() / nn::hid::VibrationNode::DefaultVibrationSampleInterval.GetMilliSeconds()) * 4;

        // ベースのBnvibファイルのMeta情報を取得します
        nn::hid::VibrationFileInfo info;
        nn::hid::VibrationFileParserContext context;
        NN_ASSERT(nn::hid::ParseVibrationFile(&info, &context, baseBnvibFileData.FileData, baseBnvibFileData.FileSize).IsSuccess());

        size_t writePosition = 0;
        {
            // Meta領域の書き込み
            uint32_t value = 0;
            // MetaDataSize
            value = g_VibrationPlayer.IsLoop() ? g_VibrationPlayer.GetLoopInterval() > 0 ? 0x00000010 : 0x0000000C : 0x00000004;
            memcpy(&outBuffer[writePosition], &value, sizeof(uint32_t));
            writePosition += 4;
            // FormatId
            value = 0x0003;
            memcpy(&outBuffer[writePosition], &value, sizeof(uint16_t));
            writePosition += 2;
            // SamplingRate
            value = 0x00C8;
            memcpy(&outBuffer[writePosition], &value, sizeof(uint16_t));
            writePosition += 2;
            // Option
            if (g_VibrationPlayer.IsLoop())
            {
                // LoopRange
                value = std::min(g_VibrationPlayer.GetFileInfo().sampleLength, static_cast<int>(g_VibrationPlayer.GetLoopStartPosition() / g_VibrationPlayer.GetPlaySpeed()));
                memcpy(&outBuffer[writePosition], &value, sizeof(uint32_t));
                writePosition += 4;
                value = std::min(g_VibrationPlayer.GetFileInfo().sampleLength, static_cast<int>(g_VibrationPlayer.GetLoopEndPosition() / g_VibrationPlayer.GetPlaySpeed()));
                memcpy(&outBuffer[writePosition], &value, sizeof(uint32_t));
                writePosition += 4;
                if (g_VibrationPlayer.GetLoopInterval() > 0)
                {
                    // LoopInterval
                    value = g_VibrationPlayer.GetLoopInterval();
                    memcpy(&outBuffer[writePosition], &value, sizeof(uint32_t));
                    writePosition += 4;
                }
            }
        }
        int32_t readPosition = 0;                    // 読み込み位置 (サンプル数基準)
        size_t metaAreaSize = writePosition;        // Meta 領域をスキップしたオフセット
        bool isEof = false;

        writePosition += 4;

        float count = 1.f - g_VibrationPlayer.GetPlaySpeed();
        auto modulation = g_VibrationTarget.Connection[0].GetModulation();

        // 再生速度による補正を加味した Data 領域のサイズ (DataSize の4byte分は含めない)
        uint32_t outDataAreaSize = 0;

        nn::hid::VibrationValue value = nn::hid::VibrationValue::Make();

        // バッファへの書き込みを実行します
        while (writePosition + 4 < BnvibFileData::FileSizeMax)
        {
            NN_ASSERT_GREATER_EQUAL(bufferSize, metaAreaSize + outDataAreaSize + 4);    // バッファが足りない場合はエラー

            if (writePosition >= metaAreaSize + offsetSize + 4)
            {
                count += g_VibrationPlayer.GetPlaySpeed();
                if (count >= 1.0f)
                {
                    if (readPosition >= info.sampleLength)
                    {
                        break;
                    }
                    for (; count >= 1.0f && readPosition < info.sampleLength; count -= 1.0f)
                    {
                        nn::hid::RetrieveVibrationValue(&value, readPosition++, &context);
                        modulation.ApplyModulation(&value);
                        isEof = (readPosition >= info.sampleLength && count >= 2.0f);
                    }
                    if (isEof)
                    {
                        break;
                    }
                }
            }
            outBuffer[writePosition++] = static_cast<uint8_t>(std::min(255.f, std::max(0.f, std::roundf(value.amplitudeLow * 255.f))));
            outBuffer[writePosition++] = static_cast<uint8_t>(std::min(255.f, std::max(0.f, std::roundf(std::log2f(value.frequencyLow / 10.f) * 32.f))));
            outBuffer[writePosition++] = static_cast<uint8_t>(std::min(255.f, std::max(0.f, std::roundf(value.amplitudeHigh * 255.f))));
            outBuffer[writePosition++] = static_cast<uint8_t>(std::min(255.f, std::max(0.f, std::roundf(std::log2f(value.frequencyHigh / 10.f) * 32.f))));
            outDataAreaSize += 4;
        }

        // データ領域のサイズを書き込みます
        outBuffer[metaAreaSize + 0] = (outDataAreaSize & 0x000000FF);
        outBuffer[metaAreaSize + 1] = (outDataAreaSize & 0x0000FF00) >> 8;
        outBuffer[metaAreaSize + 2] = (outDataAreaSize & 0x00FF0000) >> 16;
        outBuffer[metaAreaSize + 3] = (outDataAreaSize & 0xFF000000) >> 24;

        outSize = metaAreaSize + 4 + outDataAreaSize + 4;
    }
}
