﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <string>

#include "AssertDialog.h"

namespace VibrationCollection
{
    enum MountType
    {
        MountType_Sd = 0,
        MountType_Host = 1
    };

    enum PathType
    {
        PathType_Directory,
        PathType_FileName
    };

    struct LoopSetting
    {
        uint32_t isLoop;                //!< 振動ファイルがループ情報を持っていない場合は 0 が、持っている場合は 0 以外が格納されます
        uint32_t loopStartPosition;     //!< ループ開始位置
        uint32_t loopEndPosition;       //!< ループ終了位置
        uint32_t loopInterval;          //!< ループが終了してからループを開始するまでのインターバル
    };

    struct BnvibFileData
    {
        static const size_t FileSizeMax = 1024 * 1024;
        std::string         FileName;
        uint8_t*            FileData;
        size_t              FileSize;
        nn::TimeSpan        Time;
        LoopSetting         Loop;
        BnvibFileData() :
            FileData(nullptr),
            FileSize(0)
        {
            FileName = "";
            Time = nn::TimeSpan::FromMilliSeconds(0);
            Loop.isLoop = Loop.loopStartPosition = Loop.loopEndPosition = Loop.loopInterval = 0;
        }
    };

    struct AudioFileData
    {
        enum FileType
        {
            FileType_UnKnown = 0,
            FileType_Wave = 1,
            FileType_Adpcm
        };
        static const size_t         FileSizeMax = 64 * 1024 * 1024;
        std::string                 FileName;
        size_t                      FileSize;
        nn::TimeSpan                Time;
        FileType                    FileType;
        uint8_t*                    FileData;
        AudioFileData() :
            FileSize(0),
            FileType(FileType_UnKnown),
            FileData(nullptr)
        {
            FileName = "";
        }
    };

    typedef int32_t BnvibHandle;
    typedef int32_t AudioHandle;

    class FileManager
    {
        NN_DISALLOW_COPY(FileManager);
        NN_DISALLOW_MOVE(FileManager);
    public:
        static const int         MaxEntryCount = 1000;
    private:
        MountType                   m_MountType;
        std::string                 m_RootPath;
        std::string                 m_MountPath;

        std::vector<BnvibFileData>  m_BnvibFile;
        std::vector<AudioFileData>  m_AudioFile;

        MountType                   m_TempMountType;
        std::string                 m_TempRootPath;

        size_t                      m_BnvibMemorySize;
        size_t                      m_AudioMemorySize;
    public:
        ~FileManager() NN_NOEXCEPT
        {
            for (size_t i = 0; i < m_BnvibFile.size(); ++i)
            {
                if (m_BnvibFile.at(i).FileData != nullptr)
                {
                    delete[] m_BnvibFile.at(i).FileData;
                    m_BnvibFile.at(i).FileData = nullptr;
                }
            }
            for (size_t i = 0; i < m_AudioFile.size(); ++i)
            {
                if (m_AudioFile.at(i).FileData != nullptr)
                {
                    delete[] reinterpret_cast<uint8_t*>(m_AudioFile.at(i).FileData);
                    m_AudioFile.at(i).FileData = nullptr;
                }
            }
        }

        static FileManager& GetInstance() NN_NOEXCEPT
        {
            static FileManager instance;
            return instance;
        }

        nn::Result Mount() NN_NOEXCEPT;
        void UnMount() NN_NOEXCEPT;

        bool CheckPath(std::string str, PathType type) NN_NOEXCEPT;

        nn::Result ExportFile(const char* fileName, uint8_t* buffer, size_t size) NN_NOEXCEPT;

        void SetRootPath(const char* path) NN_NOEXCEPT { m_TempRootPath = path; }
        const char* GetRootPath() NN_NOEXCEPT { return m_RootPath.c_str(); }
        const char* GetTempRootPath() NN_NOEXCEPT { return m_TempRootPath.c_str(); }

        std::vector<BnvibFileData>& GetBnvibFile() NN_NOEXCEPT { return m_BnvibFile; }
        std::vector<AudioFileData>& GetAudioFile() NN_NOEXCEPT { return m_AudioFile; }

        void ChangePath() NN_NOEXCEPT;

        int LoadBnvib(bool isLoadMemory) NN_NOEXCEPT;
        int LoadAudio(bool isLoadMemory) NN_NOEXCEPT;

        void UnloadBnvib() NN_NOEXCEPT;
        void UnloadAudio() NN_NOEXCEPT;

        nn::Result ReadBnvib(BnvibHandle handle) NN_NOEXCEPT;
        nn::Result ReadAudio(AudioHandle handle) NN_NOEXCEPT;

        int GetBnvibFileCount() NN_NOEXCEPT { return m_BnvibFile.size(); }
        int GetAudioFileCount() NN_NOEXCEPT { return m_AudioFile.size(); }

        MountType GetMountType() NN_NOEXCEPT { return m_MountType; }
        MountType GetTempMountType() NN_NOEXCEPT { return m_TempMountType; }

        bool IsNeedReload() NN_NOEXCEPT { return (m_MountType != m_TempMountType) || (m_RootPath != m_TempRootPath); }

        void SetMountType(MountType type) NN_NOEXCEPT
        {
            m_TempMountType = type;
        }
    private:
        FileManager() NN_NOEXCEPT;
    };

#define gFileManager (FileManager::GetInstance())

    void Reload(void* pushButton, void* param) NN_NOEXCEPT;
}
