﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Audio.h"

namespace VibrationCollection
{
    NN_OS_ALIGNAS_THREAD_STACK char Audio::ThreadStack[Audio::UpdateThreadStackSize];

    Audio& Audio::GetInstance() NN_NOEXCEPT
    {
        static Audio instance;
        return instance;
    }

    void Audio::UpdateAudioThreadFunction(void *arg) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(arg);
        nn::os::TimerEventType* pTimerEvent = (nn::os::TimerEventType*)arg;
        while (true)
        {
            nn::os::WaitTimerEvent(pTimerEvent);
            gAudio.Update();
        }
        nn::os::StopTimerEvent(pTimerEvent);
    }

    void Audio::Initialize() NN_NOEXCEPT
    {
        // メモリプールの初期化
        m_Allocator.Initialize(WorkBufferSize);
        m_WaveBufferAllocator.Initialize(WaveBufferSize);
        m_EffectBufferAllocator.Initialize(EffectBufferSize);

        m_Voice.clear();
        m_SeVoice = VoiceData();

        // レンダラのパラメータを指定します。
        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.sampleRate = RenderRate;
        parameter.sampleCount = RenderCount;
        parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
        parameter.subMixCount = 2;
        parameter.voiceCount = 24;
        parameter.sinkCount = 1;
        parameter.effectCount = 3;
        parameter.performanceFrameCount = 0;

        // ミックスバッファとオーディオバスの関係を定義します。
        m_MainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
        m_MainBus[nn::audio::ChannelMapping_FrontRight] = 5;
        m_AuxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
        m_AuxBusA[nn::audio::ChannelMapping_FrontRight] = 1;
        m_AuxBusA[nn::audio::ChannelMapping_FrontLeft] = 2;
        m_AuxBusA[nn::audio::ChannelMapping_FrontRight] = 3;

        // パラメータがシステムでサポートされているかどうかを確認します。
        NN_ABORT_UNLESS(
            nn::audio::IsValidAudioRendererParameter(parameter),
            "Invalid AudioRendererParameter specified."
        );

        // レンダラのワークバッファを準備します。
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        void* workBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer);

        nn::os::SystemEvent systemEvent;

        // レンダラを初期化します。
        NN_ABORT_UNLESS(
            nn::audio::OpenAudioRenderer(&m_Handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
            "Failed to open AudioRenderer"
        );

        // AudioRendererConfig を初期化します。
        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        void* configBuffer = m_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(configBuffer);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, configBuffer, configBufferSize);

        nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, 6);

        for (int i = 0; i < static_cast<int>(sizeof(m_SubMix) / sizeof(m_SubMix[0])); ++i)
        {
            nn::audio::AcquireSubMix(&m_Config, &m_SubMix[i], parameter.sampleRate, 1);
        }

        // レンダラの出力先を用意します。
        nn::audio::DeviceSinkType deviceSink;
        // オーディオ出力デバイスへの入力を設定します。
        // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
        // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
        // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
        nn::Result result = nn::audio::AddDeviceSink(&m_Config, &deviceSink, &m_FinalMix, m_MainBus, ChannelCount, "MainAudioOut");
        NN_ABORT_UNLESS(result.IsSuccess());

        // SubMix(0) を FinalMix へ接続します
        nn::audio::SetSubMixDestination(&m_Config, &m_SubMix[0], &m_FinalMix);

        // SubMix(0) の 0 番目のバッファを FinalMix の mainBus[0]、mainBus[1] へ
        ///ボリューム 0.5f でミックスするように設定します
        nn::audio::SetSubMixMixVolume(&m_SubMix[0], &m_FinalMix, 0.5f, 0, m_MainBus[0]);
        nn::audio::SetSubMixMixVolume(&m_SubMix[0], &m_FinalMix, 0.5f, 0, m_MainBus[1]);

        for (int i = 1; i < static_cast<int>(sizeof(m_SubMix) / sizeof(m_SubMix[0])); ++i)
        {
            // SubMix(1) を SubMix(0) へ接続します
            nn::audio::SetSubMixDestination(&m_Config, &m_SubMix[i], &m_SubMix[0]);

            // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
            // ボリューム 0.5f でミックスするように設定します
            nn::audio::SetSubMixMixVolume(&m_SubMix[i], &m_SubMix[0], 0.5f, 0, 0);
        }

        // エフェクトバッファ用のメモリプールを準備します。
        // MemoryPool は nn::audio::AcquireMemoryPool() で渡される address 、 size の領域を書き換えることはありません。
        // そのため g_EffectBufferPoolMemory として準備した領域をすべて、メモリプールに利用することができます。
        // またこのサンプルでは WaveBuffer, Effect とメモリの用途別にアロケータを準備しています。
        // MemoryPool はその領域全体を「アタッチ」「デタッチ」します。
        // MemoryPool の操作に必要な粒度に応じてメモリ領域を分けておくことをおすすめします。
        // 詳しくは @link ../../../Samples/Sources/Applications/AudioMemoryPool
        // Samples/Sources/Applications/AudioMemoryPool @endlink を参照してください。
        nn::audio::MemoryPoolType effectBufferPool;
        bool rt = AcquireMemoryPool(&m_Config, &effectBufferPool, m_EffectBufferAllocator.GetMemoryHeap(), EffectBufferSize);
        NN_ABORT_UNLESS(rt);
        // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
        rt = RequestAttachMemoryPool(&effectBufferPool);
        NN_ABORT_UNLESS(rt);

        // Delay と BufferMixer を用意し、Delay, BufferMixer の順に登録します。
        const nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(10); // delay エフェクトで利用しうる最大のディレイ時間を指定します。
        size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, 2);
        // エフェクトバッファを取得します。g_EffectBufferAllocator が管理する全領域は effectBufferPool メモリプールに登録してあります。
        void* delayBuffer = m_EffectBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);

        nn::audio::BufferMixerType mixer0;
        nn::audio::BufferMixerType mixer1;

        //result = nn::audio::AddDelay(&m_Config, &m_Delay, delayBuffer, delaySize, &m_FinalMix, delayTimeMax, 2);
        //NN_ABORT_UNLESS(result.IsSuccess());
        //nn::audio::SetDelayInputOutput(&m_Delay, m_AuxBusA, m_AuxBusA, ChannelCount);
        //result = nn::audio::AddBufferMixer(&m_Config, &mixer1, &m_FinalMix);
        //NN_ABORT_UNLESS(result.IsSuccess());
        //nn::audio::SetBufferMixerInputOutput(&mixer1, m_AuxBusA, m_MainBus, ChannelCount);


        result = nn::audio::AddDelay(&m_Config, &m_Delay, delayBuffer, delaySize, &m_FinalMix, delayTimeMax, 2);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetDelayInputOutput(&m_Delay, m_AuxBusA, m_AuxBusA, ChannelCount);
        result = nn::audio::AddBufferMixer(&m_Config, &mixer0, &m_FinalMix);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetBufferMixerInputOutput(&mixer0, m_AuxBusB, m_AuxBusA, ChannelCount);
        result = nn::audio::AddBufferMixer(&m_Config, &mixer1, &m_FinalMix);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetBufferMixerInputOutput(&mixer1, m_AuxBusA, m_MainBus, ChannelCount);

        // Delay パラメータを設定します。
        nn::audio::SetDelayEnabled(&m_Delay, true);
        nn::audio::SetDelayTime(&m_Delay, nn::TimeSpan::FromSeconds(10));
        nn::audio::SetDelayFeedbackGain(&m_Delay, 0.7f);
        nn::audio::SetDelayChannelSpread(&m_Delay, 0.256f);
        nn::audio::SetDelayLowPassAmount(&m_Delay, 0.9f);
        nn::audio::SetDelayDryGain(&m_Delay, 0.5f);
        nn::audio::SetDelayInGain(&m_Delay, 0.4f);

        // BufferMixer パラメータを設定します。
        nn::audio::SetBufferMixerVolume(&mixer0, 0, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer0, 1, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

        // 設定したパラメータをレンダラに反映させます。
        result = nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config);
        NN_ABORT_UNLESS(result.IsSuccess());

        // レンダリングを開始します。
        result = nn::audio::StartAudioRenderer(m_Handle);
        NN_ABORT_UNLESS(result.IsSuccess());

        // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
        nn::audio::MemoryPoolType waveBufferMemoryPool;
        rt = AcquireMemoryPool(&m_Config, &waveBufferMemoryPool, m_WaveBufferAllocator.GetMemoryHeap(), WaveBufferSize);
        NN_ABORT_UNLESS(rt);
        rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
        NN_ABORT_UNLESS(rt);

        // タイマーイベントを初期化する
        nn::os::InitializeTimerEvent(&m_TimerEvent, nn::os::EventClearMode_AutoClear);
        // スレッドを生成する
        result = nn::os::CreateThread(&m_UpdateThread, UpdateAudioThreadFunction, &m_TimerEvent, ThreadStack, UpdateThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ABORT_UNLESS(rt, "Cannot create thread.");
        // スレッドの実行を開始する
        nn::os::StartThread(&m_UpdateThread);
        // タイマーイベントは周期タイマーイベントとして開始する
        nn::os::StartPeriodicTimerEvent(&m_TimerEvent, nn::TimeSpan::FromMilliSeconds(16), nn::TimeSpan::FromMilliSeconds(16));
    }

    void Audio::Finalize() NN_NOEXCEPT
    {
        m_Allocator.Finalize();
        m_EffectBufferAllocator.Finalize();
        m_WaveBufferAllocator.Finalize();
    }

    void Audio::Enable() NN_NOEXCEPT
    {
        m_IsEnable = true;
    }

    void Audio::Disable() NN_NOEXCEPT
    {
        m_IsEnable = false;
    }

    bool Audio::IsEnable() NN_NOEXCEPT
    {
        return m_IsEnable;
    }

    struct AudioAllignDataType
    {
        uint8_t data[4096];
    };

    size_t Audio::ReadAudioFile(nn::audio::AdpcmHeaderInfo* header, nns::audio::WavFormat* format, AudioHandle handle, void** data) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(format);
        NN_ASSERT_RANGE(handle, (int32_t)0, static_cast<int32_t>(gFileManager.GetAudioFile().size()));
        NN_ASSERT_NOT_NULL(data);

        size_t result = 0;
        size_t fileSize = gFileManager.GetAudioFile().at(handle).FileSize;
        switch (gFileManager.GetAudioFile().at(handle).FileType)
        {
        case AudioFileData::FileType_UnKnown:
        {
            break;
        }
        case AudioFileData::FileType_Wave:
        {
            // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
            const std::size_t WavHeaderDataSize = 1024;
            nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, gFileManager.GetAudioFile().at(handle).FileData, WavHeaderDataSize);
            if (wavResult != nns::audio::WavResult_Success)
            {
                return 0;
            }
            NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
            NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています
            result = static_cast<std::size_t>(format->dataSize);

            *data = m_WaveBufferAllocator.Allocate(static_cast<std::size_t>(result), nn::audio::BufferAlignSize);
            m_WaveBufferAllocator.Finalize();
            m_WaveBufferAllocator.Initialize(WaveBufferSize);
            *data = m_WaveBufferAllocator.Allocate(static_cast<std::size_t>(result), nn::audio::BufferAlignSize);

            std::memcpy(*data, &gFileManager.GetAudioFile().at(handle).FileData[format->dataOffset], result);
            break;
        }
        case AudioFileData::FileType_Adpcm:
        {
            NN_ASSERT_NOT_NULL(header);
            nn::audio::ParseAdpcmHeader(header, gFileManager.GetAudioFile().at(handle).FileData, nn::audio::AdpcmHeaderSize);
            result = fileSize - sizeof(nn::audio::AdpcmHeaderSize);
            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
        }
        return result;
    }

    VoiceHandle Audio::LoadWaveFile(AudioHandle handle) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(handle, (int32_t)0, static_cast<int32_t>(gFileManager.GetAudioFile().size()));

        VoiceData voice;

        nns::audio::WavFormat format;
        std::size_t dataSize = ReadAudioFile(nullptr, &format, handle, &voice.Data);

        // 読み込んだ WAV 形式ファイルのチャンネル数を指定してボイスを取得します。
        // マルチチャンネルのデータを読み込んだ場合には、チャンネル数分のボイスを消費することに注意してください。
        nn::audio::AcquireVoiceSlot(
            &m_Config, &voice.Voice, format.sampleRate,
            format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&m_Config, &voice.Voice, &m_FinalMix);

        voice.Buffer.buffer = voice.Data;
        voice.Buffer.size = dataSize;
        voice.Buffer.startSampleOffset = 0;
        voice.Buffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        voice.Buffer.loop = false;
        voice.Buffer.isEndOfStream = false;
        voice.Buffer.pContext = nullptr;
        voice.Buffer.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice.Voice, &voice.Buffer);
        nn::audio::SetVoicePlayState(&voice.Voice, nn::audio::VoiceType::PlayState_Play);
        // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
        nn::audio::SetVoiceMixVolume(&voice.Voice, &m_FinalMix, 0.f, 0, m_MainBus[0]);
        nn::audio::SetVoiceMixVolume(&voice.Voice, &m_FinalMix, 0.f, 0, m_MainBus[1]);

        // カットオフ 2048Hz のローパスフィルタを設定します
        nn::audio::BiquadFilterParameter firstFilter = { true,{ 720, 1439, 720 },{ 22684, -9350 } };
        nn::audio::SetVoiceBiquadFilterParameter(&voice.Voice, 0, firstFilter);
        // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
        nn::audio::BiquadFilterParameter secondFilter = { false,{ 14041, -28083, 14041 },{ 29547, -13563 } };
        nn::audio::SetVoiceBiquadFilterParameter(&voice.Voice, 1, secondFilter);

        m_Voice.push_back(voice);
        return m_Voice.size() - 1;
    }

    VoiceHandle Audio::LoadAdpcmFile(AudioHandle handle) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(handle, (int32_t)0, static_cast<int32_t>(gFileManager.GetAudioFile().size()));

        VoiceData voice;

        voice.Header = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(m_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAudioFile(voice.Header, nullptr, handle, nullptr);
        voice.Data = gFileManager.GetAudioFile().at(handle).FileData;

        nn::audio::AcquireVoiceSlot(
            &m_Config, &voice.Voice, voice.Header->sampleRate, 1,
            nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &voice.Header->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&m_Config, &voice.Voice, &m_FinalMix);

        voice.Buffer.buffer = voice.Data;
        voice.Buffer.size = dataSeSize;
        voice.Buffer.startSampleOffset = 0;
        voice.Buffer.endSampleOffset = voice.Header->sampleCount;
        voice.Buffer.loop = false;
        voice.Buffer.isEndOfStream = true;
        voice.Buffer.pContext = &voice.Header->loopContext;
        voice.Buffer.contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::SetVoicePlayState(&voice.Voice, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voice.Voice, &m_FinalMix, 0, 0, m_AuxBusA[0]);
        nn::audio::SetVoiceMixVolume(&voice.Voice, &m_FinalMix, 0, 0, m_AuxBusA[1]);

        nn::audio::AppendWaveBuffer(&voice.Voice, &voice.Buffer);

        m_Voice.push_back(voice);
        return m_Voice.size() - 1;
    }

    void Audio::Play(const VoiceHandle& _handle) NN_NOEXCEPT
    {
        if (IsEnable())
        {
#ifdef false
            if (_handle < 0 || _handle >= m_Voice.size())
            {
                TT_LOG("# (Aduio) 効果音の再生に失敗：ハンドル値が不正です ( 0 <= handle(%d) < %d )\n", _handle, m_Voice.size());
                return;
            }
            nn::audio::SetVoiceDestination(&m_Config, &m_Voice.at(_handle).Voice, &m_FinalMix);

            if (nn::audio::GetReleasedWaveBuffer(&m_Voice.at(_handle).Voice))
            {
                nn::audio::SetVoicePlayState(&m_Voice.at(_handle).Voice, nn::audio::VoiceType::PlayState_Play);
                nn::audio::SetVoiceMixVolume(&m_Voice.at(_handle).Voice, &m_FinalMix, 0.707f / 2, 0, m_AuxBusA[0]);
                nn::audio::SetVoiceMixVolume(&m_Voice.at(_handle).Voice, &m_FinalMix, 0.707f / 2, 0, m_AuxBusA[1]);
                nn::audio::AppendWaveBuffer(&m_Voice.at(_handle).Voice, &m_Voice.at(_handle).Buffer);
            }
#else
            NN_UNUSED(_handle);
#endif
            return;
        }
    }

    void Audio::Play(const VoiceHandle& _handle, const nn::TimeSpan& delay) NN_NOEXCEPT
    {
        AudioQueue queue;
        queue.IsWave = false;
        queue.Handle = _handle;
        queue.Delay = delay;
        m_Queue.push_back(queue);
    }

    void  Audio::PlayWav(const AudioHandle& handle) NN_NOEXCEPT
    {
        if (IsEnable())
        {
            if (nn::audio::IsVoiceValid(&m_SeVoice.Voice))
            {
                nn::audio::ReleaseVoiceSlot(&m_Config, &m_SeVoice.Voice);
            }

            nns::audio::WavFormat format;
            ApplicationHeap heap;
            std::size_t dataSize = 0;

            switch (gFileManager.GetAudioFile().at(handle).FileType)
            {
            case AudioFileData::FileType_UnKnown:
                return;
                break;
            case AudioFileData::FileType_Wave:
            {
                dataSize = ReadAudioFile(nullptr, &format, handle, &m_SeVoice.Data);
                if (dataSize == 0)
                {
                    m_SeVoice = VoiceData();
                    return;
                }
                nn::audio::AcquireVoiceSlot(
                    &m_Config, &m_SeVoice.Voice, format.sampleRate,
                    format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
                m_SeVoice.Buffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
                m_SeVoice.Buffer.pContext = nullptr;
                m_SeVoice.Buffer.contextSize = 0;
                break;
            }
            case AudioFileData::FileType_Adpcm:
            {
                return;
                break;
            }
            default: NN_UNEXPECTED_DEFAULT;
            }

            nn::audio::SetVoiceDestination(&m_Config, &m_SeVoice.Voice, &m_FinalMix);
            m_SeVoice.Buffer.buffer = m_SeVoice.Data;
            m_SeVoice.Buffer.size = dataSize;
            m_SeVoice.Buffer.startSampleOffset = 0;
            m_SeVoice.Buffer.loop = false;
            m_SeVoice.Buffer.isEndOfStream = true;

            nn::audio::AppendWaveBuffer(&m_SeVoice.Voice, &m_SeVoice.Buffer);
            nn::audio::SetVoicePlayState(&m_SeVoice.Voice, nn::audio::VoiceType::PlayState_Play);
            // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
            nn::audio::SetVoiceMixVolume(&m_SeVoice.Voice, &m_FinalMix, 1.000f, 0, m_MainBus[0]);
            nn::audio::SetVoiceMixVolume(&m_SeVoice.Voice, &m_FinalMix, 1.000f, 0, m_MainBus[1]);

            // カットオフ 2048Hz のローパスフィルタを設定します
            nn::audio::BiquadFilterParameter firstFilter = { true,{ 720, 1439, 720 },{ 22684, -9350 } };
            nn::audio::SetVoiceBiquadFilterParameter(&m_SeVoice.Voice, 0, firstFilter);
            // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
            nn::audio::BiquadFilterParameter secondFilter = { false,{ 14041, -28083, 14041 },{ 29547, -13563 } };
            nn::audio::SetVoiceBiquadFilterParameter(&m_SeVoice.Voice, 1, secondFilter);

        }
    }

    void Audio::PlayWav(const VoiceHandle& handle, const nn::TimeSpan& delay) NN_NOEXCEPT
    {
        AudioQueue queue;
        queue.IsWave = true;
        queue.Handle = handle;
        queue.Delay = delay;
        m_Queue.push_back(queue);
    }

    nn::Result Audio::Update() NN_NOEXCEPT
    {
        nn::Result result = nn::ResultSuccess();

        if (m_Queue.size() > 0)
        {
            for (
                std::vector<AudioQueue>::iterator it = m_Queue.begin();
                it != m_Queue.end();
                )
            {
                (*it).Delay -= nn::TimeSpan::FromMilliSeconds(16);
                if ((*it).Delay < nn::TimeSpan::FromMilliSeconds(8))
                {
                    (*it).IsWave ? PlayWav((*it).Handle) : Play((*it).Handle);
                    it = m_Queue.erase(it);
                }
                else
                {
                    ++it;
                }
            }
        }
        if (GetRendererState() == nn::audio::AudioRendererState::AudioRendererState_Started)
        {
            result = nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config);
            NN_ABORT_UNLESS(result.IsSuccess());
        }
        return result;
    }

    void Audio::ReleaceVoice(const VoiceHandle& _handle) NN_NOEXCEPT
    {
        if (nn::audio::IsVoiceValid(&m_Voice.at(_handle).Voice))
        {
            nn::audio::ReleaseVoiceSlot(&m_Config, &m_Voice.at(_handle).Voice);
        }
    }

    void Audio::ReLoadVoice(const VoiceHandle& _handle) NN_NOEXCEPT
    {
        NN_UNUSED(_handle);
    }

    nn::audio::AudioRendererState Audio::GetRendererState() NN_NOEXCEPT
    {
        return nn::audio::GetAudioRendererState(m_Handle);
    }

    nn::audio::VoiceType::PlayState Audio::GetSeVoiceState() NN_NOEXCEPT
    {
        if (!nn::audio::IsVoiceValid(&m_SeVoice.Voice))
        {
            return nn::audio::VoiceType::PlayState::PlayState_Stop;
        }
        if (nn::audio::GetVoicePlayedSampleCount(&m_SeVoice.Voice) == 0)
        {
            nn::audio::SetVoicePlayState(&m_SeVoice.Voice, nn::audio::VoiceType::PlayState_Stop);
        }
        return nn::audio::GetVoicePlayState(&m_SeVoice.Voice);
    }
}
