﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>

#include "FirmwareImage.h"

namespace nns { namespace hid { namespace util {

/**
 * @brief   SD カード内のファームウェアイメージを扱うクラスです。
 */
class FirmwareImageEnumerator final
{
    NN_DISALLOW_COPY(FirmwareImageEnumerator);
    NN_DISALLOW_MOVE(FirmwareImageEnumerator);

public:
    static FirmwareImageEnumerator* GetInstance() NN_NOEXCEPT;

    /**
     * @brief   SD カードがマウントされているか
     */
    bool IsMounted() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        return m_IsMounted;
    }

    /**
     * @brief   SD カードにアクセス中かどうか確認します。
     */
    bool IsBusy() const NN_NOEXCEPT;

    /**
     * @brief   SD カードをマウントします。
     */
    void Mount() NN_NOEXCEPT;

    /**
     * @brief   SD カードのマウントを解除します。
     */
    void Unmount() NN_NOEXCEPT;

    /**
     * @brief   SD カードの自動マウントを試みます。
     *
     * @return  自動マウント/アンマウントが行われた場合は true
     */
    bool TryAutoMount() NN_NOEXCEPT;

    /**
     * @brief   認識されたファイルの数を取得します。
     */
    int GetFileCount() const NN_NOEXCEPT;

    /**
     * @brief   ファイル一覧が空かどうか判定します。
     */
    bool IsEmpty() const NN_NOEXCEPT;

    /**
     * @brief   指定された番号のイメージを取得します。
     *
     * @pre
     *  - 0 <= index && index < GetFileCount()
     */
    const FirmwareImage& GetImage(int index) const NN_NOEXCEPT;

private:
    FirmwareImageEnumerator() NN_NOEXCEPT;

    ~FirmwareImageEnumerator() NN_NOEXCEPT;

    void ClearFileList() NN_NOEXCEPT;

    void ListupFiles() NN_NOEXCEPT;

private:
    static const int FileCountMax = 256;        //!< 列挙できるファイルの最大数

private:
    mutable nn::os::Mutex   m_Mutex;

    bool            m_IsMounted;                //!< マウント済みか
    bool            m_IsSdCardDetected;         //!< SD カードが検出されているか
    FirmwareImage   m_Images[FileCountMax];     //!< イメージファイル一覧
    int             m_FileCount;                //!< イメージファイル数
};

}}}  // nns::hid::util
