﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/time.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_ScopeExit.h>

#define NNS_HID_UTIL_ENABLE_TIMESTAMP

namespace nns { namespace hid { namespace util {

/**
 * @brief   イメージに対応するデバイスです。
 */
struct DeviceType
{
    typedef nn::util::BitFlagSet<64, DeviceType>::Flag<0>   JoyLeft;    //!< Joy-Con (L)
    typedef nn::util::BitFlagSet<64, DeviceType>::Flag<1>   JoyRight;   //!< Joy-Con (R)
    typedef nn::util::BitFlagSet<64, DeviceType>::Flag<2>   FullKey;    //!< Pro Controller
};

typedef nn::util::BitFlagSet<64, DeviceType> DeviceTypeSet;

const char DeviceTypeNameJoyLeft[]  = "JoyLeft";
const char DeviceTypeNameJoyRight[] = "JoyRight";
const char DeviceTypeNameFullKey[]  = "FullKey";
const char DeviceTypeNameProCon[]   = "ProCon";

/**
 * @brief   イメージに対応するチップです。
 */
enum class ChipType
{
    Mcu1,
    Mcu2
};

const char ChipTypeNameMcu1[] = "Mcu1";
const char ChipTypeNameMcu2[] = "Mcu2";

/**
 * @brief   イメージファイルの基本情報です。
 */
struct FirmwareImageInfo
{
    char                    name[nn::fs::EntryNameLengthMax];   //!< ファイル名
    int64_t                 size;                               //!< ファイルサイズ

#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
    nn::fs::FileTimeStamp   timeStamp;                          //!< 更新日時
#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)

    DeviceTypeSet           device;                             //!< 対応するデバイス
    ChipType                chip;                               //!< 対応するチップ
    char                    version[32];                        //!< バージョン情報

    void Clear() NN_NOEXCEPT
    {
        name[0]    = '0';
        size       = 0;

#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
        timeStamp  = nn::fs::FileTimeStamp();
#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)

        device.Reset();
        chip       = ChipType::Mcu1;
        version[0] = '0';
    }
};

/**
 * @brief   ファームウェアイメージと対応デバイスの情報を扱うクラスです。
 */
class FirmwareImage final
{
public:
    /**
     * @brief   ファームウェアイメージを格納しているディレクトリを設定します。
     */
    static void SetBaseDirectory(const char* directory) NN_NOEXCEPT;

    FirmwareImage() NN_NOEXCEPT;

    /**
     * @brief   格納している情報をクリアします。
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   ファームウェアイメージ情報を設定します。
     */
    void SetInfo(const FirmwareImageInfo& info) NN_NOEXCEPT;

    /**
     * @brief   ファームウェアイメージのファイル名を取得します。
     */
    const char* GetFileName() const NN_NOEXCEPT
    {
        return m_Info.name;
    }

    /**
     * @brief   ファームウェアイメージのサイズを取得します。
     */
    int64_t GetFileSize() const NN_NOEXCEPT
    {
        return m_Info.size;
    }

    /**
     * @brief   ファームウェアバージョンを取得します。
     */
    const char* GetVersion() const NN_NOEXCEPT
    {
        return m_Info.version;
    }

#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
    /**
     * @brief   ファームウェアイメージの最終更新日時を取得します。
     */
    nn::fs::FileTimeStamp GetTimeStamp() const NN_NOEXCEPT
    {
        return m_Info.timeStamp;
    }

    /**
     * @brief   ファームウェアイメージの最終更新日時を取得します。
     */
    nn::time::CalendarTime GetModifiedTime() const NN_NOEXCEPT
    {
        nn::time::CalendarTime time;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::time::ToCalendarTime(&time, nullptr, m_Info.timeStamp.modify)
        );
        return time;
    }

#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)

    /**
     * @brief   ファームウェアの対応デバイスを取得します。
     */
    DeviceTypeSet GetTargetDeviceType() const NN_NOEXCEPT
    {
        return m_Info.device;
    }

    /**
     * @brief   ファームウェアの対応デバイス名を取得します。
     */
    const char* GetTargetDeviceName() const NN_NOEXCEPT
    {
        return m_TargetName;
    }

    /**
     * @brief   ファームウェアの対応チップ名を取得します。
     */
    const char* GetTargetChipName() const NN_NOEXCEPT;

    /**
     * @brief   ファームウェアの対応チップを取得します。
     */
    nn::hid::debug::FirmwareUpdateTargetChip GetTargetChip() const NN_NOEXCEPT;

    /**
     * @brief   ファームウェアイメージを読み込みます。
     */
    nn::Result ReadImage(char* pOutImageData, size_t dataSizeMax) const NN_NOEXCEPT;

private:
    static const int TargetNameLengthMax = 32;

private:
    void CreateTargetName() NN_NOEXCEPT;

    void GetFullPath(char* pOutPath, size_t pathLengthMax) const NN_NOEXCEPT;

private:
    static char         m_BaseDirectory[nn::fs::EntryNameLengthMax];

    FirmwareImageInfo   m_Info;
    char                m_TargetName[TargetNameLengthMax];
};

}}}  // nns::hid::util
