﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <string>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "FirmwareImage.h"
#include "util.h"

namespace nns { namespace hid { namespace util {

namespace
{

}  // anonymous

char FirmwareImage::m_BaseDirectory[nn::fs::EntryNameLengthMax] = {};

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR;

void FirmwareImage::SetBaseDirectory(const char* directory) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(directory);

    nn::util::Strlcpy(m_BaseDirectory, directory, sizeof(m_BaseDirectory));
}

FirmwareImage::FirmwareImage() NN_NOEXCEPT
    : m_Info()
    , m_TargetName()
{
}

void FirmwareImage::Clear() NN_NOEXCEPT
{
    m_Info.Clear();
    CreateTargetName();
}

void FirmwareImage::SetInfo(const FirmwareImageInfo& info) NN_NOEXCEPT
{
    m_Info = info;
    CreateTargetName();
}

const char* FirmwareImage::GetTargetChipName() const NN_NOEXCEPT
{
    switch (m_Info.chip)
    {
    case ChipType::Mcu1:
        return "Bluetooth";

    case ChipType::Mcu2:
        return "MCU";

    default:
        return "Unknown";
    }
}

nn::hid::debug::FirmwareUpdateTargetChip FirmwareImage::GetTargetChip() const NN_NOEXCEPT
{
    switch (m_Info.chip)
    {
    case ChipType::Mcu1:
        return nn::hid::debug::FirmwareUpdateTargetChip_Bluetooth;

    case ChipType::Mcu2:
        return nn::hid::debug::FirmwareUpdateTargetChip_Mcu;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result FirmwareImage::ReadImage(char* pOutImageData, size_t dataSizeMax) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImageData);
    NN_ASSERT_GREATER_EQUAL(dataSizeMax, GetFileSize());

    char path[nn::fs::EntryNameLengthMax];
    GetFullPath(path, sizeof(path));

    nn::fs::FileHandle fileHandle;
    NN_RESULT_DO(
        nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read)
    );

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(fileHandle);
    };

    NN_RESULT_DO(
        nn::fs::ReadFile(fileHandle, 0, pOutImageData, GetFileSize())
    );

    NN_RESULT_SUCCESS;
}

void FirmwareImage::CreateTargetName() NN_NOEXCEPT
{
    std::string name;

    auto appendName = [&name](const char* text)
    {
        if (!name.empty())
        {
            name.append("/");
        }

        name.append(text);
    };

    if (m_Info.device.Test<DeviceType::JoyLeft>())
    {
        appendName("JoyLeft");
    }

    if (m_Info.device.Test<DeviceType::JoyRight>())
    {
        appendName("JoyRight");
    }

    if (m_Info.device.Test<DeviceType::FullKey>())
    {
        appendName("ProCon");
    }

    if (name.empty())
    {
        appendName("Unknown");
    }

    nn::util::Strlcpy(m_TargetName, name.c_str(), sizeof(m_TargetName));
}

void FirmwareImage::GetFullPath(char* pOutPath, size_t pathLengthMax) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutPath);
    NN_ASSERT_GREATER(pathLengthMax, 0);

    nn::util::SNPrintf(pOutPath, pathLengthMax, "%s/%s", m_BaseDirectory, GetFileName());
}

}}}  // nns::hid::util
