﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../util/Config.h"
#include "../util/FirmwareImage.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "UniquePadButton.h"

namespace nns { namespace hid { namespace ui {

namespace
{

util::DeviceTypeSet GetDeviceTypeSet(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    switch (nn::hid::system::GetUniquePadType(id))
    {
    case nn::hid::system::UniquePadType_LeftController:
        return util::DeviceType::JoyLeft::Mask;

    case nn::hid::system::UniquePadType_RightController:
        return util::DeviceType::JoyRight::Mask;

    case nn::hid::system::UniquePadType_FullKeyController:
        return util::DeviceType::FullKey::Mask;

    default:
        return util::DeviceTypeSet();
    }
}

}  // anonymous

UniquePadButton::UniquePadButton() NN_NOEXCEPT
    : m_pManager(nullptr)
    , m_Id()
    , m_Index(0)
    , m_TargetDeviceType()
{
    SetButtonMaskForPress(util::DecideButtonMask);
}

void UniquePadButton::SetManager(util::UniquePadManager* pManager) NN_NOEXCEPT
{
    m_pManager = pManager;
}

void UniquePadButton::SetUniquePadId(const nn::hid::system::UniquePadId& id, int index) NN_NOEXCEPT
{
    m_Id    = id;
    m_Index = index;
}

void UniquePadButton::SetTargetDevice(util::DeviceTypeSet deviceType) NN_NOEXCEPT
{
    m_TargetDeviceType = deviceType;
}

void UniquePadButton::Update() NN_NOEXCEPT
{
    auto& config = *util::Config::GetInstance();
    if (config.IsForceUpdateEnabled())
    {
        SetEnabled(true);
    }
    else
    {
        auto type = GetDeviceTypeSet(m_Id);
        SetEnabled((type & m_TargetDeviceType).IsAnyOn());
    }

    Button::Update();
}

void UniquePadButton::RenderContents() NN_NOEXCEPT
{
    if (m_pManager == nullptr)
    {
        return;
    }

    const auto& clientSize = GetClientAreaSize();

    nns::sgx::ScopedFontContextSaver saver;

    nns::sgx::SetTextSize(24);
    nns::sgx::SetTextColor(nns::sgx::Colors::Shadow());

    auto& str = nns::hid::util::StringTable::GetInstance();

    // デバイス名
    nns::sgx::DrawText(12, 16, "%s", util::GetUniquePadTypeName(m_Id));

    // インジケーター
    nn::hid::system::UniquePadInterface padInterface;
    if (nn::hid::system::GetUniquePadInterface(&padInterface, m_Id).IsSuccess())
    {
        nns::sgx::ScopedFontContextSaver saverForIndicator;

        if (padInterface == nn::hid::system::UniquePadInterface_Rail)
        {
            nns::sgx::SetTextColor(nns::sgx::Colors::Smoke());
            nns::sgx::SetTextSize(16);
            nns::sgx::DrawText(0, 0, "OnRail");
        }
        else
        {
            auto pattern = util::GetControllerIndicatorPattern(m_Id);
            util::DrawControllerIndicator(2, 4, pattern, 10);
        }
    }

    // FW バージョン
    const float versionWidth = 200;
    char versionText[100];
    nn::hid::system::FirmwareVersion version;
    if (m_pManager->GetFirmwareVersion(&version, m_Index))
    {
        nn::util::SNPrintf(
            versionText,
            sizeof(versionText),
            str.Get("FlashWriter_VersionTextFormat"),
            version.identifier,
            version.major,
            version.minor,
            version.micro,
            version.revision
        );
    }
    else
    {
        nn::util::Strlcpy(versionText, str.Get("FlashWriter_VersionReading"), sizeof(versionText));
    }
    nns::sgx::SetTextSize(20);
    nns::sgx::SetTextColor(nns::sgx::Colors::DimGray());
    nns::sgx::DrawText(
        clientSize.width - versionWidth,
        14,
        versionWidth,
        24,
        nns::sgx::TextAlignment::Right,
        "%s",
        versionText
    );
}

}}}  // nns::hid::ui
