﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/SimpleGfx_PostEffect.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../util/FirmwareImage.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "FirmwareInfoView.h"

namespace nns { namespace hid { namespace ui {

namespace
{

const int AnimationFrameCount = 5;

}  // anonymous

FirmwareInfoView::FirmwareInfoView() NN_NOEXCEPT
    : m_pImage(nullptr)
    , m_AnimDuration(0)
    , m_CurrentAnimation(AnimationType::None)
    , m_OriginX(0)
{
}

void FirmwareInfoView::SetImage(const util::FirmwareImage* pImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);

    m_pImage = pImage;
}

void FirmwareInfoView::Appear() NN_NOEXCEPT
{
    ForceFinishAnimation();

    m_AnimDuration     = AnimationFrameCount;
    m_CurrentAnimation = AnimationType::Appear;
    m_OriginX          = GetX();
}

void FirmwareInfoView::Disappear() NN_NOEXCEPT
{
    ForceFinishAnimation();

    m_AnimDuration     = AnimationFrameCount;
    m_CurrentAnimation = AnimationType::Disappear;
    m_OriginX          = GetX();
}

void FirmwareInfoView::Update() NN_NOEXCEPT
{
    Canvas::Update();

    UpdateAnimation();
}

void FirmwareInfoView::RenderContents() NN_NOEXCEPT
{
    if (m_pImage == nullptr)
    {
        return;
    }

    const int opacity = GetDisplayOpacity();
    if (opacity <= 0)
    {
        return;
    }

    const float offset = 2.0f * opacity / 255.0f;
    nns::sgx::ApplyBlurEffect(offset, offset);

    nns::sgx::FillRectangle(0, 0, GetWidth(), GetHeight(), nns::sgx::Colors::Shadow().BlendAlpha(192 * opacity / 255));

#if 0
    nns::sgx::DrawGradientLine(
        3, 3, 3, GetHeight() / 2,
        nns::sgx::Colors::WhiteSmoke(),
        nns::sgx::Colors::LightGray(),
        4
    );
    nns::sgx::DrawGradientLine(
        3, GetHeight() / 2, 3, GetHeight(),
        nns::sgx::Colors::LightGray(),
        nns::sgx::Colors::WhiteSmoke(),
        4
    );
#endif

    nns::sgx::ScopedFontContextSaver saver;

    auto& str = nns::hid::util::StringTable::GetInstance();
    const float captionOffset = 92.0f;

    float dx = 24.0f;
    float dy = 10.0f;
    nns::sgx::SetTextSize(26);
    nns::sgx::SetTextColor(nns::sgx::Colors::PaleGreen().BlendAlpha(opacity));
    nns::sgx::DrawText(dx, dy, str.Get("FlashWriter_CaptionTargetDevice"));
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, str.Get("FlashWriter_CaptionTargetChip"));
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, str.Get("FlashWriter_CaptionFirmwareVersion"));
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, str.Get("FlashWriter_CaptionFileSize"));
#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, str.Get("FlashWriter_CaptionModifiedTimeStamp"));
#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)

    dx += 16.0f;
    dy  = 48.0f;
    nns::sgx::SetTextSize(24);
    nns::sgx::SetTextColor(nns::sgx::Colors::WhiteSmoke().BlendAlpha(opacity));
    nns::sgx::DrawText(dx, dy, m_pImage->GetTargetDeviceName());
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, m_pImage->GetTargetChipName());
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, m_pImage->GetVersion());
    dy += captionOffset;
    nns::sgx::DrawText(dx, dy, "%.2f KB", m_pImage->GetFileSize() / 1024.0f);
#if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
    dy += captionOffset;
    const auto& time = m_pImage->GetModifiedTime();
    nns::sgx::DrawText(dx, dy, "%04d/%02d/%02d %02d:%02d:%02d",
        time.year,
        time.month,
        time.day,
        time.hour,
        time.minute,
        time.second
    );
#endif  // if defined(NNS_HID_UTIL_ENABLE_TIMESTAMP)
}

void FirmwareInfoView::UpdateAnimation() NN_NOEXCEPT
{
    if (!IsAnimating())
    {
        return;
    }

    m_AnimDuration--;

    float progress = static_cast<float>(AnimationFrameCount - m_AnimDuration) / AnimationFrameCount;
    switch (m_CurrentAnimation)
    {
    case AnimationType::Appear:
        {
            SetOpacity(static_cast<uint8_t>(255 * progress));
            SetPosition(m_OriginX + 16 * (1 - progress), GetY());
        }
        break;

    case AnimationType::Disappear:
        {
            SetOpacity(static_cast<uint8_t>(255 * (1 - progress)));
            SetPosition(m_OriginX + 16 * progress, GetY());
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (m_AnimDuration <= 0)
    {
        m_CurrentAnimation = AnimationType::None;
        SetPosition(m_OriginX, GetY());
    }
}

void FirmwareInfoView::ForceFinishAnimation() NN_NOEXCEPT
{
    while (IsAnimating())
    {
        UpdateAnimation();
    }
}

}}}  // nns::hid::ui
