﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   ウィンドウキャプションの最大長です。
 */
const int WindowTextLengthMax = 64;

/**
 * @brief   ウィンドウのパディング初期値です。
 */
const float WindowPaddingDefault = 4.0f;

/**
 * @brief   ウィンドウの状態です。
 */
enum WindowState
{
    WindowState_Init,       //!< 初期状態
    WindowState_Idle        //!< 通常
};

/**
 * @brief   ウィンドウの配色です。
 */
struct WindowColorScheme
{
    Color border;                   //!< 枠線
    Color contentsBackgroundBegin;  //!< コンテンツ背景始点
    Color contentsBackgroundEnd;    //!< コンテンツ背景終点
    Color captionBackground;        //!< キャプション背景
    Color captionText;              //!< キャプション文字
};

/**
 * @brief   ウィンドウを扱うクラスです。
 */
class Window :
    public UiContainer
{
    NN_DISALLOW_COPY(Window);
    NN_DISALLOW_MOVE(Window);

public:
    /**
     * @brief   アクティブ時のデフォルトの配色を設定します。
     */
    static void SetDefaultActiveColor(const WindowColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultActiveColor = scheme;
    }

    /**
     * @brief   非アクティブ時のデフォルトの配色を設定します。
     */
    static void SetDefaultInactiveColor(const WindowColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultInactiveColor = scheme;
    }

public:
    Window() NN_NOEXCEPT :
        m_State(WindowState_Init),
        m_Caption(),
        m_CaptionHeight(36.0f),
        m_TextSize(40.0f),
        m_ActiveColor(),
        m_InactiveColor()
    {}

    /**
     * @brief   クライアント領域の矩形を取得します。
     */
    virtual Rectangle GetClientArea() const NN_NOEXCEPT NN_OVERRIDE
    {
        auto rect = UiContainer::GetClientArea();
        rect.y      += m_CaptionHeight;
        rect.height -= m_CaptionHeight;

        return rect;
    }

    /**
     * @brief   ドラッグを認識する領域を取得します。
     */
    virtual void GetDraggableArea(Rectangle* pOutArea) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT_NOT_NULL(pOutArea);
        NNS_SGX_GUI_SCOPED_LOCK;

        GetAbsolutePosition(&pOutArea->position);
        pOutArea->size.width  = GetWidth();
        pOutArea->size.height = m_CaptionHeight;
    }

    /**
     * @brief   キャプション領域の高さを設定します。
     */
    void SetCaptionHeight(float height) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_CaptionHeight = std::max(height, 0.0f);
    }

    /**
     * @brief   アクティブ時の配色を設定します。
     */
    void SetActiveColor(const WindowColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_ActiveColor = scheme;
    }

    /**
     * @brief   非アクティブ時の配色を設定します。
     */
    void SetInactiveColor(const WindowColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_InactiveColor = scheme;
    }

    /**
     * @brief   キャプションを UTF-8 で設定します。
     */
    void SetText(const char* text) NN_NOEXCEPT;

    /**
     * @brief   キャプションを UTF-32 で設定します。
     */
    void SetText(const uint32_t* text) NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    /**
     * @brief   ウィンドウの枠と背景を描画します。
     */
    virtual void RenderBackground() NN_NOEXCEPT;

private:
    /**
     * @brief   デフォルトの配色を初期化します。
     */
    static void InitializeDefaultColorScheme() NN_NOEXCEPT;

    /**
     * @brief   配色を初期化します。
     */
    void SetupColorScheme() NN_NOEXCEPT;

    /**
     * @brief   必要に応じて初期化処理を行います。
     *
     * @note    静的初期化時に配色が初期化されていない可能性があるため、
     *          最初に更新または描画されるまで初期化を遅延します。
     */
    void InitializeIfNeeded() NN_NOEXCEPT
    {
        if (m_State == WindowState_Init)
        {
            SetupColorScheme();
            m_State = WindowState_Idle;
        }
    }

private:
    static bool                 g_IsFirstInitialized;
    static WindowColorScheme    g_DefaultActiveColor;
    static WindowColorScheme    g_DefaultInactiveColor;
    static std::vector<Window>  g_WindowList;

private:
    WindowState         m_State;
    uint32_t            m_Caption[WindowTextLengthMax];
    float               m_CaptionHeight;
    float               m_TextSize;
    WindowColorScheme   m_ActiveColor;
    WindowColorScheme   m_InactiveColor;
};

}}}  // nns::sgx::gui
