﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   UI 要素のレイアウト方向です。
 */
enum class LayoutDirection
{
    Vertical,   //!< 縦方向
    Horizontal  //!< 縦方向
};

/**
 * @brief   UI 要素の整列方法です。
 */
enum class LayoutAlignment
{
    Left,               //!< 左寄せ
    Center,             //!< 中央寄せ
    Right,              //!< 右寄せ
    Top    = Left,      //!< 上寄せ
    Middle = Center,    //!< 中央寄せ
    Bottom = Right      //!< 下寄せ
};

/**
 * @brief   縦または横方向に要素を自動的に並べるレイアウトコンテナです。
 */
class FlowLayoutContainer : public UiContainer
{
    NN_DISALLOW_COPY(FlowLayoutContainer);
    NN_DISALLOW_MOVE(FlowLayoutContainer);

public:
    FlowLayoutContainer() NN_NOEXCEPT;

    /**
     * @brief   子要素のレイアウト方向を取得します。
     */
    LayoutDirection GetDirection() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Direction;
    }

    /**
     * @brief   子要素のレイアウト方向を設定します。
     */
    void SetDirection(LayoutDirection direction) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        if (m_Direction != direction)
        {
            m_Direction = direction;
            UpdateLayout();
        }
    }

    /**
     * @brief   子要素の整列方法を取得します。
     */
    LayoutAlignment GetAlignment() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Alignment;
    }

    /**
     * @brief   子要素の整列方法方向を設定します。
     *          @ref IsAutoWrap() == false の場合のみ使用されます。
     */
    void SetAlignment(LayoutAlignment alignment) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        if (m_Alignment != alignment)
        {
            m_Alignment = alignment;
            UpdateLayout();
        }
    }

    /**
     * @brief   子要素の配置間隔を設定します。
     */
    void SetItemGap(const Point2D& gap) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_ItemGap = gap;
    }

    /**
     * @brief   子要素の自動改行を行うかどうか取得します。
     */
    virtual bool IsAutoWrap() NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_IsAutoWrap;
    }

    /**
     * @brief   子要素の自動改行を行うかどうか設定します。
     */
    virtual void SetAutoWrap(bool isAutoWrap) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        if (m_IsAutoWrap != isAutoWrap)
        {
            m_IsAutoWrap = isAutoWrap;
            UpdateLayout();
        }
    }

    /**
     * @brief   コンテナに子要素を追加します。
     *
     * @details コンテナに子要素を追加し、自動でレイアウトが行われるようにします。
     *          追加された子要素は、手動で表示位置を調整しないようにしてください。
     *          子要素のサイズを変更した場合は、 @ref UpdateLayout を呼び出して再配置してください。
     */
    virtual void AddChild(DisplayObject* pChild) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   コンテナから指定した要素を削除します。
     */
    virtual void RemoveChild(const DisplayObject* pChild) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   レイアウトの更新を行い、子要素を再配置します。
     */
    void UpdateLayout() NN_NOEXCEPT;

private:
    LayoutDirection     m_Direction;    //!< レイアウト方向
    LayoutAlignment     m_Alignment;    //!< 整列方法
    Point2D             m_ItemGap;      //!< 子要素の配置間隔
    bool                m_IsAutoWrap;   //!< 自動改行
};

}}}  // nns::sgx::gui
