﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiLabel.h"

namespace nns { namespace sgx { namespace gui {

Label::Label() NN_NOEXCEPT
    : m_Text()
    , m_TextSize(40.0f)
    , m_Color(Colors::White())
    , m_ShadowColor(Colors::Shadow())
    , m_BackgroundColor(Colors::Transparent())
    , m_Padding()
    , m_IsShadowEnabled(false)
{
}

void Label::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    nn::util::ConvertStringUtf8ToUtf32(m_Text, NN_ARRAY_SIZE(m_Text), text);
}

void Label::SetText(const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    for (int i = 0; i < NN_ARRAY_SIZE(m_Text) - 1 && text[i] != 0u; i++)
    {
        m_Text[i] = text[i];
    }
}

void Label::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    // 表示サイズが設定されている場合は描画範囲を制限
    bool isClientAreaSpecified = GetWidth() > 0 && GetHeight() > 0;
    if (isClientAreaSpecified)
    {
        Rectangle rect;
        GetRenderPosition(&rect.position);
        rect.size = GetSize();
        NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect);
        nns::sgx::ApplyRenderArea(rect);

        // 背景色が指定されていれば背景を描画
        if (m_BackgroundColor.GetA() > 0)
        {
            nns::sgx::FillRectangle(0, 0, rect.width, rect.height, m_BackgroundColor);
        }
    }

    nns::sgx::ScopedFontContextSaver saver;

    nns::sgx::SetTextSize(m_TextSize);

    float x = m_Padding.x + (isClientAreaSpecified ? 0 : GetX());
    float y = m_Padding.y + (isClientAreaSpecified ? 0 : GetY());

    if (IsShadowEnabled())
    {
        nns::sgx::SetTextColor(m_ShadowColor);
        nns::sgx::DrawText(x + 1, y + 1, m_Text);
    }

    nns::sgx::SetTextColor(m_Color);
    nns::sgx::DrawText(x, y, m_Text);

    if (isClientAreaSpecified)
    {
        nns::sgx::RestoreRenderArea();
    }
}

}}}  // nns::sgx::gui
