﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_NpadCommonTypes.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"
#include "SimpleGfx_GuiButton.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   ダイアログの Z オーダー初期値です。
 */
const int DefaultDialogZ = 15000;

/**
 * @brief   ダイアログ表示を行うクラスです。
 */
class Dialog :
    public UiContainer
{
    NN_DISALLOW_COPY(Dialog);
    NN_DISALLOW_MOVE(Dialog);

public:
    Dialog() NN_NOEXCEPT;

    virtual void SetSize(const Size& size) NN_NOEXCEPT NN_OVERRIDE
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        UiContainer::SetSize(size);
        Centering();
        UpdateLayout();
    }

    virtual void SetSize(float width, float height) NN_NOEXCEPT NN_OVERRIDE
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        UiContainer::SetSize(width, height);
        Centering();
        UpdateLayout();
    }

    virtual void SetVisible(bool isVisible) NN_NOEXCEPT NN_OVERRIDE
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        UiContainer::SetVisible(isVisible);

        if (!isVisible)
        {
            SetFocusedChild(nullptr);
        }
    }

    /**
     * @brief   ダイアログの状態を初期化します。
     *
     * @details ダイアログを操作する前に必ず呼び出す必要があります。
     */
    virtual void Initialize() NN_NOEXCEPT;

    /**
     * @brief   テキスト領域のサイズを取得します。
     */
    virtual Size GetTextAreaSize() const NN_NOEXCEPT;

    /**
     * @brief   テキスト領域の矩形を取得します。
     */
    virtual Rectangle GetTextArea() const NN_NOEXCEPT;

    /**
     * @brief   ボタンの最大数を取得します。
     */
    int GetButtonCountMax() const NN_NOEXCEPT
    {
        return NN_ARRAY_SIZE(m_Buttons);
    }

    /**
     * @brief   ボタンの数を取得します。
     */
    int GetButtonCount() const NN_NOEXCEPT
    {
        return m_ButtonCount;
    }

    /**
     * @brief   ボタンの数を設定します。
     *
     * @details ボタンの数を変更すると、必ず先頭のボタンにフォーカスが当たっている状態になります。
     *          2 番目以降のボタンにフォーカスを当てたい場合は、別途 @ref SetFocusedButton を呼ぶ必要があります。
     */
    void SetButtonCount(int count) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_MINMAX(count, 0, GetButtonCountMax());

        NNS_SGX_GUI_SCOPED_LOCK;

        if (m_ButtonCount == count)
        {
            return;
        }

        m_ButtonCount = count;

        if (count > 0)
        {
            SetFocusedButton(0);
        }

        UpdateLayout();
    }

    /**
     * @brief   ボタンのテキストを UTF-8 で設定します。
     */
    void SetButtonText(int buttonIndex, const char* text) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        m_Buttons[buttonIndex].SetText(text);
    }

    /**
     * @brief   ボタンのテキストを UTF-32 で設定します。
     */
    void SetButtonText(int buttonIndex, const uint32_t* text) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        NNS_SGX_GUI_SCOPED_LOCK;

        m_Buttons[buttonIndex].SetText(text);
    }

    /**
     * @brief   指定したボタンの有効状態を設定します。
     */
    void SetButtonEnabled(int buttonIndex, bool isEnabled) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        m_Buttons[buttonIndex].SetEnabled(isEnabled);
    }

    /**
     * @brief   全ボタンのスキンを一括で設定します。
     */
    void SetButtonSkin(const ButtonSkin& skin) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);

        NNS_SGX_GUI_SCOPED_LOCK;

        for (auto& button : m_Buttons)
        {
            button.SetSkin(skin);
        }
    }

    /**
     * @brief   指定したボタンのスキンを設定します。
     */
    void SetButtonSkin(int buttonIndex, const ButtonSkin& skin) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        NNS_SGX_GUI_SCOPED_LOCK;

        m_Buttons[buttonIndex].SetSkin(skin);
    }

    /**
     * @brief   フォーカスが当たっているボタンを設定します。
     */
    void SetFocusedButton(int buttonIndex) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        NNS_SGX_GUI_SCOPED_LOCK;

        SetFocusedChild(&m_Buttons[buttonIndex]);
    }

    /**
     * @brief   指定したボタンの押下判定をする Npad のボタンを設定します。
     */
    void SetButtonMaskForPress(int buttonIndex, nn::hid::NpadButtonSet mask) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        NNS_SGX_GUI_SCOPED_LOCK;

        m_Buttons[buttonIndex].SetButtonMaskForPress(mask);
    }

    /**
     * @brief   キャンセル判定をする Npad のボタンを設定します。
     */
    void SetButtonMaskForCancel(nn::hid::NpadButtonSet mask) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);

        NNS_SGX_GUI_SCOPED_LOCK;

        m_CancelButtonMask = mask;
    }

    /**
     * @brief   指定したボタン押下時のイベントハンドラを設定します。
     *
     * @details ユーザー引数には常に 0 が渡されます。
     */
    void SetButtonPushEventHandler(int buttonIndex, GuiEventHandler handler) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);

        NNS_SGX_GUI_SCOPED_LOCK;

        SetButtonPushEventHandler(buttonIndex, handler, 0);
    }

    /**
     * @brief   指定したボタン押下時のイベントハンドラを設定します。
     */
    void SetButtonPushEventHandler(int buttonIndex, GuiEventHandler handler, uintptr_t userArg) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT_RANGE(buttonIndex, 0, GetButtonCount());

        NNS_SGX_GUI_SCOPED_LOCK;

        m_Buttons[buttonIndex].SetPushEventHandler(handler, userArg);
    }

    /**
     * @brief   ダイアログを表示します。
     */
    virtual void Appear() NN_NOEXCEPT;

    /**
     * @brief   ダイアログを非表示にします。
     */
    virtual void Disappear() NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   キャンセル操作で呼び出されるハンドラを登録します。
     *
     * @details ユーザー引数には常に 0 が渡されます。
     */
    void SetCancelHandler(const GuiEventHandler& handler) NN_NOEXCEPT
    {
        SetCancelHandler(handler, 0);
    }

    /**
     * @brief   キャンセル操作で呼び出されるハンドラを登録します。
     */
    void SetCancelHandler(const GuiEventHandler& handler, uintptr_t argument) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForCancel.handler  = handler;
        m_HandlerForCancel.argument = argument;
    }

    /**
     * @brief   テキスト領域の描画タイミングで呼び出されるハンドラを登録します。
     *
     * @details ユーザー引数には常に 0 が渡されます。
     */
    void SetRenderTextAreaHandler(const GuiEventHandler& handler) NN_NOEXCEPT
    {
        SetRenderTextAreaHandler(handler, 0);
    }

    /**
     * @brief   テキスト領域の描画タイミングで呼び出されるハンドラを登録します。
     *
     * @details 登録したハンドラは、RenderTextArea の後に呼び出されます。
     *          新たなクラスを定義せずに、一部の描画のみ変更したい場合に使用できます。
     */
    void SetRenderTextAreaHandler(const GuiEventHandler& handler, uintptr_t argument) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForRenderTextArea.handler  = handler;
        m_HandlerForRenderTextArea.argument = argument;
    }

    virtual bool UpdateTouchInput() NN_NOEXCEPT NN_OVERRIDE
    {
        UiContainer::UpdateTouchInput();

        // アクティブ状態の場合、下にある UI のタッチ操作は受け付けない
        return IsVisible() && IsEnabled() && IsFocused();
    }

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   テキスト領域を描画します。継承先で上書きして使用します。
     */
    virtual void RenderTextArea() NN_NOEXCEPT {}

    /**
     * @brief   UI パーツを自動配置します。
     */
    void UpdateLayout() NN_NOEXCEPT;

    /**
     * @brief   ダイアログの上下左右中央にテキストを描画します。
     */
    void DrawCenteringText(const char* text) NN_NOEXCEPT;

    /**
     * @brief   ダイアログの左右中央にテキストを描画します。
     */
    void DrawCenteringText(float y, const char* text) NN_NOEXCEPT;

private:
    /**
     * @brief   ダイアログの表示状態です。
     */
    enum class DisplayState
    {
        Hide,       //!< 非表示
        Hiding,     //!< 非表示アニメ中
        Showing,    //!< 表示アニメ中
        Show        //!< 表示
    };

private:

    /**
     * @brief   ボタンにフォーカスが当たっていない場合、先頭のボタンをフォーカス状態にします。
     */
    void SetDefaultFocus() NN_NOEXCEPT;

    /**
     * @brief   表示位置を画面中央に移動します。
     */
    void Centering() NN_NOEXCEPT;

    /**
     * @brief   アニメーションを更新します。
     */
    void UpdateAnimation() NN_NOEXCEPT;

private:
    bool                    m_IsInitialized;
    int                     m_ButtonCount;
    Button                  m_Buttons[3];
    ButtonSkin              m_ButtonSkin;
    nn::hid::NpadButtonSet  m_CancelButtonMask;
    GuiEventHandlerType     m_HandlerForCancel;         //!< キャンセル操作ハンドラ
    GuiEventHandlerType     m_HandlerForRenderTextArea; //!< テキスト領域の描画ハンドラ

    DisplayState            m_DisplayState;
    int                     m_AnimDuration;
};

}}}  // nns::sgx::gui
