﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiCircleProgressBar.h"

namespace nns { namespace sgx { namespace gui {

CircleProgressBar::~CircleProgressBar() NN_NOEXCEPT
{
    if (m_GradImage.IsValid())
    {
        nns::sgx::DestroyImage(&m_GradImage);
    }
}

void CircleProgressBar::Update() NN_NOEXCEPT
{
    ProgressBar::Update();

    m_Duration++;
}

void CircleProgressBar::DrawBar() NN_NOEXCEPT
{
    if (!m_GradImage.IsValid())
    {
        // TODO: ファイル指定対応
        nns::sgx::LoadImage(&m_GradImage, "rom:/CircleGradGreen.png");
    }

    auto opacity = GetDisplayOpacity();
    auto rate    = GetDispRate();
    auto angle   = 2.0f * nn::util::FloatPi * rate;
    auto radius  = std::min(GetSize().width, GetSize().height) / 2.0f;

    nns::sgx::SetStencilEnabled(true);
    nns::sgx::ClearStencil();

    // 円形のゲージ領域のみ描画対象 (Stencil 1) にする
    nns::sgx::BeginRenderToStencil();
    nns::sgx::SetStencilFunc(nns::sgx::StencilFunc::Always, 1, ~0);
    nns::sgx::FillCircle(radius, radius, radius, nns::sgx::Colors::White());
    nns::sgx::SetStencilFunc(nns::sgx::StencilFunc::Always, 0, ~0);
    nns::sgx::FillCircle(radius, radius, radius - m_BarWidth, nns::sgx::Colors::White());
    nns::sgx::EndRenderToStencil();

    // ゲージ背景を描画
    nns::sgx::SetStencilFunc(nns::sgx::StencilFunc::Equal, 1, ~0);
    FillRectangle(0, 0, GetWidth(), GetHeight(), Colors::Smoke().BlendAlpha(opacity));

    if (rate > 0.0f)
    {
        // ゲージ未到達部分を描画範囲から除外 (Stencil 0)
        const auto initAngle = -nn::util::FloatPi / 2.0f;
        nns::sgx::BeginRenderToStencil();
        nns::sgx::SetStencilFunc(nns::sgx::StencilFunc::Always, 0, ~0);
        FillPie(
            radius,
            radius,
            radius,
            initAngle + angle,
            initAngle + 2.0f * nn::util::FloatPi,
            Colors::White());
        nns::sgx::EndRenderToStencil();

        // ゲージ内部を描画
        nns::sgx::SetStencilFunc(nns::sgx::StencilFunc::Equal, 1, ~0);
        nns::sgx::SetImageOpacity(opacity / 255.0f);
        nns::sgx::SetImageRotation(m_Duration * nn::util::FloatPi / 45.0f);
        nns::sgx::DrawImage(m_GradImage, 0, 0, GetWidth(), GetHeight());
        nns::sgx::SetImageRotation(0);
        nns::sgx::SetImageOpacity(1.0f);
    }

    nns::sgx::SetStencilEnabled(false);
}

}}}  // nns::sgx::gui
