﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_NpadCommonTypes.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   ボタンテキストの最大長
 */
const int ButtonTextLengthMax = 64;

/**
 * @brief   ボタンの見た目です。
 */
enum class ButtonAppearance
{
    Normal   = 0,   //!< 通常
    Focused  = 1,   //!< フォーカス状態
    Pressed  = 2,   //!< 押下状態
    Disabled = 3    //!< 無効状態
};

/**
 * @brief   ボタンを扱うベースクラスです。
 */
class ButtonBase :
    public DisplayObject
{
    NN_DISALLOW_COPY(ButtonBase);
    NN_DISALLOW_MOVE(ButtonBase);

public:
    ButtonBase() NN_NOEXCEPT;

    /**
     * @brief   キー操作によってフォーカス可能かどうか取得します。
     */
    virtual bool IsKeyFocusable() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    /**
     * @brief   ボタン押下として扱う Npad のボタンを設定します。
     */
    void SetButtonMaskForPress(nn::hid::NpadButtonSet mask) NN_NOEXCEPT
    {
        m_ButtonMaskForPress = mask;
    }

    /**
     * @brief   ボタン押下時の処理を設定します。
     *
     * @details ハンドラのユーザー引数には常に 0 が渡されます。
     */
    void SetPushEventHandler(GuiEventHandler handler) NN_NOEXCEPT
    {
        SetPushEventHandler(handler, 0);
    }

    /**
     * @brief   ボタン押下時の処理を設定します。
     */
    void SetPushEventHandler(GuiEventHandler handler, uintptr_t userArg) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForPushEvent.handler  = handler;
        m_HandlerForPushEvent.argument = userArg;
    }

    /**
     * @brief   ボタンが押下されているか判定します。
     */
    bool IsPressed() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        auto state = GetState();
        return state == ButtonState_PressByKey || state == ButtonState_PressByTouch;
    }

    /**
     * @brief   ボタンの見た目を取得します。
     */
    ButtonAppearance GetAppearance() const NN_NOEXCEPT;

    /**
     * @brief   ボタン押下時の処理を実行します。
     */
    void PerformPress() NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateKeyInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void CancelTouchInput() NN_NOEXCEPT NN_OVERRIDE;

protected:
    /**
     * @brief   ボタンの状態です。
     */
    enum ButtonState
    {
        ButtonState_Init,           //!< 初期状態
        ButtonState_Idle,           //!< 通常
        ButtonState_PressByTouch,   //!< タッチによる押下状態
        ButtonState_PressByKey,     //!< キー操作による押下状態
    };

protected:
    /**
     * @brief   ボタンの状態を取得します。
     */
    ButtonState GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    /**
     * @brief   ボタンの状態を設定します。
     */
    void SetState(ButtonState state) NN_NOEXCEPT
    {
        m_State = state;
    }

private:
    ButtonState                 m_State;
    nn::hid::NpadButtonSet      m_ButtonMaskForPress;
    GuiEventHandlerType         m_HandlerForPushEvent;
};

}}}  // nns::sgx::gui
