﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <cctype>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include <nn/pl.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_Util.h"
#include "SimpleGfx_Font.h"

namespace nns { namespace sgx { namespace detail {

namespace
{

// フォントの種類
enum FontType
{
    FontType_Standard,
    FontType_NintendoExtension,
    FontType_Max
};

// フォントの探索範囲
struct FontRange
{
    uint32_t begin;
    uint32_t end;
};

// フォントの探索範囲
const FontRange FontSearchRange[] =
{
    { 0x00000, 0x09FFF },  // 0 - CJK
    { 0x0E000, 0x0E153 },  // Nintendo extension
    //{ 0x20000, 0x2A6DF },  // CJK-B
};

// 任天堂外字の範囲
const FontRange NintendoExtensionRange[] =
{
    //{ 0x25A0, 0x25A1 },  // 代替文字
    { 0xE000, 0xE099 },  // 旧プラットフォーム
    { 0xE0A0, 0xE153 }   // Switch
};

// フォントエンジンのワークバッファサイズ
const size_t FontEngineWorkSize = 8 * 1024 * 1024;

const int CharCountMax = 0x10000;
const int LineCharCount = 94;

const float FontCharSpace = 1;
const float FontLineSpace = 4;

const int TexturePitch = 1;

// 描画に必要なグリフ情報
struct GlyphInfo
{
    bool    isValid;
    int16_t loX;
    int16_t hiY;
    int16_t idX;
    int16_t idY;
    int16_t width;
    int16_t height;

    void Setup(const nn::fontll::GlyphMap& glyph) NN_NOEXCEPT
    {
        isValid = false;
        loX     = glyph.loX;
        hiY     = glyph.hiY;
        idX     = glyph.idX;
        idY     = glyph.idY;
        width   = glyph.width;
        height  = glyph.height;
    }
};

// フォントの情報
struct FontInfo
{
    char                name[nn::fontll::FontNameLengthMax];
    int                 totalCharCount;
    nn::fontll::Metrics metrics[FontType_Max];

    float               charSpace;
    float               lineSpace;
    int                 maxBoundWidth;
    int                 maxHeight;
    int                 baseLineY;
    float               ascentRatio;
    Point2D             gridOffset;
    int*                pFontIndexTable;
    GlyphInfo*          pGlyphCache;

    GLuint              textureId;
    int                 textureGridWidth;
    int                 textureGridHeight;
    int                 textureWidth;
    int                 textureHeight;
    char*               textureBuffer;

    // メンバを初期化
    void Initialize() NN_NOEXCEPT
    {
        totalCharCount  = 0;
        std::memset(metrics, 0, sizeof(metrics));

        charSpace       = FontCharSpace;
        lineSpace       = FontLineSpace;
        maxBoundWidth   = 0;
        maxHeight       = 0;
        baseLineY       = 0;
        ascentRatio     = 1.0f;
        gridOffset      = Point2D();
        pFontIndexTable = nullptr;
        pGlyphCache     = nullptr;

        textureId           = 0;
        textureGridWidth    = 0;
        textureGridHeight   = 0;
        textureWidth        = 0;
        textureHeight       = 0;
        textureBuffer       = nullptr;
    }
};

// 初期化済みフラグ
bool g_IsInitialized = false;

// 初期化パラメータ
FontInitializationParameters g_Params = {};

// スケーラブルフォントのエンジン
nn::fontll::ScalableFontEngine g_Engines[FontType_Max];

// スケーラブルフォントエンジンのワークバッファ
char* g_pFontEngineWorkBuffer[FontType_Max] = {};

// フォント全体の情報
FontInfo g_FontInfo;

// 指定した文字に対応するフォントタイプを取得
FontType GetCharacterFontType(uint32_t character) NN_NOEXCEPT
{
    for (const auto& range : NintendoExtensionRange)
    {
        if (character >= range.begin && character <= range.end)
        {
            return FontType_NintendoExtension;
        }
    }

    return FontType_Standard;
}

// 指定した文字に対応するグリフを取得
GlyphInfo* GetGlyph(FontType type, uint32_t character) NN_NOEXCEPT
{
    NN_ASSERT_LESS(type, FontType_Max);

    auto index = g_FontInfo.pFontIndexTable[character];
    if (index >= 0)
    {
        // キャッシュ済みのグリフを返す
        return &g_FontInfo.pGlyphCache[index];
    }

    auto engine = g_Engines[type];
    if (!engine.CheckGlyphExist(character))
    {
        // 指定した文字のグリフは存在しない
        return nullptr;
    }

    auto pGlyph = engine.AcquireGlyphmap(character, nn::fontll::FormatGrayMap8);
    if (pGlyph == nullptr)
    {
        return nullptr;
    }

    NN_UTIL_SCOPE_EXIT
    {
        engine.ReleasesGlyph(pGlyph);
    };

    // グリフ情報をキャッシュ
    auto& glyph = g_FontInfo.pGlyphCache[g_FontInfo.totalCharCount];
    glyph.Setup(*pGlyph);

    // フォント情報の更新
    g_FontInfo.maxBoundWidth = std::max<int>(g_FontInfo.maxBoundWidth, pGlyph->idX);
    g_FontInfo.maxHeight     = std::max<int>(g_FontInfo.maxHeight, pGlyph->height);
    g_FontInfo.gridOffset.x  = g_FontInfo.maxBoundWidth;
    g_FontInfo.gridOffset.y  = g_FontInfo.maxHeight + g_FontInfo.lineSpace;

    g_FontInfo.pFontIndexTable[character] = g_FontInfo.totalCharCount;
    g_FontInfo.totalCharCount++;

    return &glyph;
}

// 指定した文字をテクスチャに描画
void DrawGlyphToTexture(FontType type, uint32_t character) NN_NOEXCEPT
{
    NN_ASSERT_LESS(type, FontType_Max);
    NN_ASSERT_LESS(character, CharCountMax);

    auto pGlyphInfo = GetGlyph(type, character);
    if (pGlyphInfo == nullptr ||
        pGlyphInfo->isValid)
    {
        return;
    }

    auto engine = g_Engines[type];
    if (!engine.CheckGlyphExist(character))
    {
        return;
    }

    auto pGlyph = engine.AcquireGlyphmap(character, nn::fontll::FormatGrayMap8);
    if (pGlyph == nullptr)
    {
        return;
    }

    // グリフを処理済みにする
    pGlyphInfo->isValid = true;

    if (g_Params.isDynamicMode)
    {
        // テクスチャの更新準備
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, g_FontInfo.textureId);
    }

    // グリフイメージの描画
    auto index = g_FontInfo.pFontIndexTable[character];
    int offsetFromTop = std::max(g_FontInfo.baseLineY - pGlyph->hiY, 0);
    int startX = (index % LineCharCount) * g_FontInfo.textureGridWidth;
    int startY = (index / LineCharCount) * g_FontInfo.textureGridHeight + offsetFromTop;
    for (int y = 0; y < pGlyph->height; y++)
    {
        int offset = (startY + y) * g_FontInfo.textureWidth + startX + pGlyph->loX;
        for (int x = 0; x < pGlyph->width; x++)
        {
            int texturePosition = (offset + x) * TexturePitch;
            int bitPosition = y * pGlyph->width + x;

            // グリフの形状をアルファ値として格納
            g_FontInfo.textureBuffer[texturePosition] = pGlyph->bits[bitPosition];

#if 0
            // 枠線を描画
            if (x == 0 || x == pGlyph->width - 1 || y == 0 || y == pGlyph->height - 1)
            {
                g_FontInfo.textureBuffer[texturePosition] |= 0x80;
            }
#endif
        }

        if (g_Params.isDynamicMode)
        {
            // テクスチャの更新
            glTexSubImage2D(
                GL_TEXTURE_2D,
                0,
                startX + pGlyph->loX,
                startY + y,
                static_cast<GLsizei>(pGlyph->width),
                1u,
                GL_RED,
                GL_UNSIGNED_BYTE,
                &g_FontInfo.textureBuffer[offset * TexturePitch]);
        }
    }

    engine.ReleasesGlyph(pGlyph);
}

// フォントエンジンの初期化
void InitializeFontEngine() NN_NOEXCEPT
{
    auto startTick = nn::os::GetSystemTick();

    const nn::pl::SharedFontType sharedFontTypes[] =
    {
        nn::pl::SharedFontType_Standard,
        nn::pl::SharedFontType_NintendoExtension
    };

    for (int i = 0; i < FontType_Max; i++)
    {
        auto type = sharedFontTypes[i];
        while (nn::pl::GetSharedFontLoadState(type) != nn::pl::SharedFontLoadState_Loaded)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }

        // フォントのバイナリデータを取得
        //auto fontBinarySize = nn::pl::GetSharedFontSize(type);
        auto scalableFontBinary = nn::pl::GetSharedFontAddress(type);

        g_pFontEngineWorkBuffer[i] = new char[FontEngineWorkSize];
        NN_ABORT_UNLESS_NOT_NULL(g_pFontEngineWorkBuffer[i]);

        // フォントの初期化
        auto& engine = g_Engines[i];
        engine.Initialize(g_pFontEngineWorkBuffer[i], FontEngineWorkSize);
        engine.LoadFont(g_FontInfo.name, scalableFontBinary, 0, nn::fontll::FontNameLengthMax);
        engine.SetFont(g_FontInfo.name);

        engine.SetScale(g_Params.baseFontSize << 16, 0, 0, g_Params.baseFontSize << 16);
    }

    PrintDiffTime(NN_CURRENT_FUNCTION_NAME, startTick);
}

// フォント情報の初期化
void InitializeFontInfo() NN_NOEXCEPT
{
    auto startTick = nn::os::GetSystemTick();

    // フォントの metrics を取得
    for (int i = 0; i < FontType_Max; i++)
    {
        g_Engines[i].GetFontMetrics(&g_FontInfo.metrics[i]);
    }

    // ベースライン位置を決める
    g_FontInfo.ascentRatio =
        static_cast<float>(g_FontInfo.metrics[0].metricsResolution) / g_FontInfo.metrics[0].os2WinAscent;
        //static_cast<float>(g_FontInfo.metrics[0].os2WinAscent) / g_FontInfo.metrics[0].metricsResolution;
    g_FontInfo.baseLineY = static_cast<int>(g_FontInfo.ascentRatio * g_Params.baseFontSize + 0.5f);

    // グリフキャッシュのための情報を生成
    g_FontInfo.pFontIndexTable = new int[CharCountMax];
    g_FontInfo.pGlyphCache     = new GlyphInfo[CharCountMax];
    NN_ABORT_UNLESS_NOT_NULL(g_FontInfo.pFontIndexTable);
    NN_ABORT_UNLESS_NOT_NULL(g_FontInfo.pGlyphCache);

    for (int i = 0; i < CharCountMax; i++)
    {
        g_FontInfo.pFontIndexTable[i] = -1;
    }
    std::memset(g_FontInfo.pGlyphCache, 0, sizeof(GlyphInfo) * CharCountMax);

    if (g_Params.isDynamicMode)
    {
        // 動的モードでは、グリフを事前スキャンしないので、多少の冗長性は許容してほどほどの値にする

        // 最大サイズはフォントサイズ 32 の値を基準として仮決め
        g_FontInfo.maxBoundWidth = g_Params.baseFontSize * 37 / 32;
        g_FontInfo.maxHeight     = g_Params.baseFontSize * 41 / 32;

        // テクスチャグリッドは大きめ
        g_FontInfo.textureGridWidth  = g_Params.baseFontSize * 3 / 2;
        g_FontInfo.textureGridHeight = g_FontInfo.textureGridWidth;

        // テクスチャサイズは 2^n でないとおかしくなる
        g_FontInfo.textureWidth  = RoundUpPow2(LineCharCount * g_FontInfo.textureGridWidth);
        g_FontInfo.textureHeight = g_FontInfo.textureWidth;
    }
    else
    {
        // 静的モードでは、全グリフをスキャンして必要十分な値を事前に決める

        int charCount     = 0;
        int maxBoundWidth = 0;
        int maxHeight     = 0;

        // 指定の FontRange 内のグリフをスキャン
        auto scanRange = [&](
            nn::fontll::ScalableFontEngine* pEngine,
            const FontRange& range) NN_NOEXCEPT
        {
            for (int i = range.begin; i <= range.end; i++)
            {
                if (!pEngine->CheckGlyphExist(i))
                {
                    // 指定した文字のグリフは存在しない
                    continue;
                }

                if (g_FontInfo.pFontIndexTable[i] >= 0)
                {
                    // 既にグリフ確認済み
                    continue;
                }

                auto pGlyph = pEngine->AcquireGlyphmap(i, nn::fontll::FormatGrayMap8);
                if (pGlyph == nullptr)
                {
                    continue;
                }

                // グリフ情報をキャッシュ
                auto& glyph = g_FontInfo.pGlyphCache[charCount];
                glyph.Setup(*pGlyph);

                maxBoundWidth = std::max<int>(maxBoundWidth, pGlyph->idX);
                maxHeight     = std::max<int>(maxHeight, pGlyph->height);

                g_FontInfo.pFontIndexTable[i] = charCount;
                charCount++;
                NN_ASSERT_LESS(charCount, CharCountMax);

                pEngine->ReleasesGlyph(pGlyph);
            }
        };

        for (auto& engine : g_Engines)
        {
            for (const auto& range : FontSearchRange)
            {
                scanRange(&engine, range);
            }
        }

        g_FontInfo.totalCharCount = charCount;

        g_FontInfo.maxBoundWidth     = maxBoundWidth;
        g_FontInfo.maxHeight         = maxHeight;
        g_FontInfo.textureGridWidth  = maxBoundWidth;
        g_FontInfo.textureGridHeight = maxHeight;

        // テクスチャサイズは 2^n でないとおかしくなる
        g_FontInfo.textureWidth  = RoundUpPow2(
            LineCharCount * g_FontInfo.textureGridWidth);
        g_FontInfo.textureHeight = RoundUpPow2(
            ((g_FontInfo.totalCharCount - 1) / LineCharCount + 1) * g_FontInfo.textureGridHeight);
    }

    g_FontInfo.gridOffset.x = g_FontInfo.maxBoundWidth;
    g_FontInfo.gridOffset.y = g_FontInfo.maxHeight + g_FontInfo.lineSpace;

    PrintDiffTime(NN_CURRENT_FUNCTION_NAME, startTick);

}  // NOLINT(readability/fn_size)

// フォントテクスチャの初期化
void InitializeTexture() NN_NOEXCEPT
{
    auto startTick = nn::os::GetSystemTick();

    size_t textureSize = g_FontInfo.textureWidth * g_FontInfo.textureHeight * TexturePitch;
    g_FontInfo.textureBuffer = new char[textureSize];
    NN_ABORT_UNLESS_NOT_NULL(g_FontInfo.textureBuffer);

    std::memset(g_FontInfo.textureBuffer, 0, textureSize);

    if (!g_Params.isDynamicMode)
    {
        // 静的モードでは最初に全グリフイメージを生成
        const FontType fontTypes[] = { FontType_Standard, FontType_NintendoExtension };
        for (auto type : fontTypes)
        {
            for (const auto& range : FontSearchRange)
            {
                for (uint32_t i = range.begin; i <= range.end; i++)
                {
                    DrawGlyphToTexture(type, i);
                }
            }
        }
    }

    // テクスチャを生成
    glGenTextures(1, &g_FontInfo.textureId);

    // Red をアルファ値として扱う
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, g_FontInfo.textureId);
    glTexImage2D(
        GL_TEXTURE_2D,
        0,
        GL_R8,
        static_cast<GLsizei>(g_FontInfo.textureWidth),
        static_cast<GLsizei>(g_FontInfo.textureHeight),
        0,  // border: Always 0
        GL_RED,
        GL_UNSIGNED_BYTE,
        g_FontInfo.textureBuffer
    );

    const GLint swizzleMask[] = { GL_ONE, GL_ONE, GL_ONE, GL_RED };
    glTexParameteriv(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzleMask);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    PrintDiffTime(NN_CURRENT_FUNCTION_NAME, startTick);
}

}  // anonymous

void InitializeSharedFont(const FontInitializationParameters& param) NN_NOEXCEPT
{
    if (g_IsInitialized)
    {
        return;
    }

    g_Params = param;

    g_FontInfo.Initialize();

    InitializeFontEngine();
    InitializeFontInfo();
    InitializeTexture();

    g_IsInitialized = true;
}

void FinalizeSharedFont() NN_NOEXCEPT
{
    if (!g_IsInitialized)
    {
        return;
    }

    glDeleteTextures(1, &g_FontInfo.textureId);
    for (auto& engine : g_Engines)
    {
        engine.Finalize();
    }

    if (g_FontInfo.pFontIndexTable != nullptr)
    {
        delete[] g_FontInfo.pFontIndexTable;
        g_FontInfo.pFontIndexTable = nullptr;
    }

    if (g_FontInfo.pGlyphCache != nullptr)
    {
        delete[] g_FontInfo.pGlyphCache;
        g_FontInfo.pGlyphCache = nullptr;
    }

    if (g_FontInfo.textureBuffer != nullptr)
    {
        delete[] g_FontInfo.textureBuffer;
        g_FontInfo.textureBuffer = nullptr;
    }

    for (int i = 0; i < FontType_Max; i++)
    {
        if (g_pFontEngineWorkBuffer[i] != nullptr)
        {
            delete[] g_pFontEngineWorkBuffer[i];
            g_pFontEngineWorkBuffer[i] = nullptr;
        }
    }

    g_IsInitialized = false;
}

int GetSharedFontBaseSize() NN_NOEXCEPT
{
    NN_ASSERT(g_IsInitialized, "Not initialized");

    return g_Params.baseFontSize;
}

GLuint GetSharedFontTextureId() NN_NOEXCEPT
{
    NN_ASSERT(g_IsInitialized, "Not initialized");

    return g_FontInfo.textureId;
}

float GetSharedFontLineHeight() NN_NOEXCEPT
{
    NN_ASSERT(g_IsInitialized, "Not initialized");

    return g_FontInfo.gridOffset.y;
}

bool IsValidSharedFontCharacter(uint32_t character) NN_NOEXCEPT
{
    return character < CharCountMax;
}

void GetSharedFontGridSize(Size* pOutSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);

    pOutSize->width  = g_FontInfo.maxBoundWidth;
    pOutSize->height = g_FontInfo.maxHeight;
}

void GetSharedFontCharacterSize(Size* pOutSize, uint32_t character) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);

    auto pGlyph = GetGlyph(GetCharacterFontType(character), character);
    if (pGlyph == nullptr)
    {
        pOutSize->width  = 0;
        pOutSize->height = 0;
        return;
    }

    // 幅は可変、高さは固定
    pOutSize->width  = pGlyph->idX;
    pOutSize->height = g_FontInfo.maxHeight;
}

void GetSharedFontTextureCoord(Point2D* pOutCoord, uint32_t character) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutCoord);

    auto type   = GetCharacterFontType(character);
    auto pGlyph = GetGlyph(type, character);
    if (pGlyph == nullptr)
    {
        return;
    }
    else if (!pGlyph->isValid)
    {
        DrawGlyphToTexture(type, character);
    }

    auto index = g_FontInfo.pFontIndexTable[character];
    auto baseX = (index % LineCharCount) * g_FontInfo.textureGridWidth;
    auto baseY = (index / LineCharCount) * g_FontInfo.textureGridHeight;

    // ピクセル値をテクスチャ座標に変換
    auto x      = static_cast<float>(baseX) / g_FontInfo.textureWidth;
    auto y      = static_cast<float>(baseY) / g_FontInfo.textureHeight;
    auto width  = static_cast<float>(pGlyph->idX) / g_FontInfo.textureWidth;
    auto height = static_cast<float>(g_FontInfo.maxHeight) / g_FontInfo.textureHeight;

    pOutCoord[0].x = x;
    pOutCoord[0].y = y;
    pOutCoord[1].x = x + width;
    pOutCoord[1].y = y;
    pOutCoord[2].x = x + width;
    pOutCoord[2].y = y + height;
    pOutCoord[3].x = x;
    pOutCoord[3].y = y + height;
}

}}}  // nns::sgx::detail
