﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       折れ線グラフ作成のためのAPI宣言
 */

#pragma once

#include "layout_Base.h"

namespace nns { namespace hidfw { namespace layout {

    //----------------------------------------------------------------
    //! @brief 縦棒グラフを作成するためのクラスです
    //----------------------------------------------------------------
    class LineChart : public BaseItem
    {
    public:
        //----------------------------------------------------------------
        //! @brief グラフの要素
        //----------------------------------------------------------------
        struct Element
        {
            int                 ValueIndex;                     //!< 最新の値のインデックス
            nn::util::Color4u8  Color;                          //!< 要素の色
            std::string         Name;                           //!< 要素の名前
            float               MaxValue;                       //!< 値の最大値 (要素毎)
            float               MinValue;                       //!< 値の最小値 (要素毎)
            float               LineWidth;                      //!< 線の太さ
            std::vector<float>  Value;                          //!< 値
            bool                Visible;                        //!< 表示するか否か
            Element() NN_NOEXCEPT :
                ValueIndex(0),
                Color(nn::util::Color4u8::White()),
                Name(""),
                MaxValue(0.f),
                MinValue(0.f),
                LineWidth(1.f),
                Visible(false)
            {
                Value.resize(0);
            }
        };

    public:
        static const int        MaxElementCount = 16;           //!< グラフに追加できる要素数の最大値
        static const int        MaxValueLength = 1024;         //!< 一つの要素が持てる値の最大長

    public:

        LineChart() NN_NOEXCEPT;
        virtual ~LineChart() NN_NOEXCEPT {}
        virtual void Update() NN_NOEXCEPT;
        virtual void PrintText() NN_NOEXCEPT;
        virtual void Draw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  デフォルト値を設定
        //----------------------------------------------------------------
        virtual void SetDefault() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      枠の表示を残す
        //! @details    非表示設定時も枠の描画のみ有効化します
        //! @param[in]  enable trueの時、枠の表示を残す
        //----------------------------------------------------------------
        void LeaveFrameDrawing(bool enable) NN_NOEXCEPT { m_IsFrameDraw = enable; }

        //----------------------------------------------------------------
        //! @brief      要素に値を追加する
        //! @param[in]  index   対象の要素のインデックス
        //! @param[in]  value   追加する値
        //! @pre        index < 要素数
        //----------------------------------------------------------------
        void AddValue(int index, float value) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      要素を追加する
        //! @details    入力された要素はコピーして使用されます
        //! @param[in]  data    追加する要素
        //----------------------------------------------------------------
        int AddElement(const Element& data) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      要素を取得する
        //! @return     追加された要素
        //----------------------------------------------------------------
        const std::vector<Element>& GetElements() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      特定の要素の表示状態を変更します
        //! @param[in]  index   対象の要素のインデックス
        //! @param[in]  enable  true の場合非表示
        //! @pre        index < 要素数
        //----------------------------------------------------------------
        bool VisibleElement(int index, bool enable) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      要素を作成します
        //! @param[in]  color       要素の色
        //! @param[in]  name        要素の名前
        //! @param[in]  valueCount  要素数
        //! @param[in]  maxValue    要素が持つ値の最大値
        //! @param[in]  minValue    要素が持つ値の最小値
        //! @param[in]  lineWidth   要素の線の太さ
        //! @return                 作成された要素
        //----------------------------------------------------------------
        static Element CreateElement(nn::util::Color4u8 color, const char* name, int valueCount, float maxValue, float minValue, float lineWidth = 1.f) NN_NOEXCEPT;

    protected:
        bool                    m_IsFrameDraw;                  //!< true の場合、Visibleがfalseであっても枠のみ描画する
        std::vector<Element>    m_Elements;                     //!< 要素リスト
    };

}}}
