﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "layout_Dialog.h"

namespace nns { namespace hidfw { namespace layout {

    Button Dialog::DialogButton[MaxDialogButtonCount];
    bool Dialog::IsInitialized = false;
    int Dialog::OpenDialogCount = 0;

    Dialog::Dialog() NN_NOEXCEPT :
        m_IsExit(false),
        m_DialogResult(DialogResult_Ignore),
        m_ExitCount(0)
    {
        if (!IsInitialized)
        {
            // 未初期化状態ならボタンを初期化します
            static DialogParam dialogParam[MaxDialogButtonCount];
            for (int i = 0; i < MaxDialogButtonCount; ++i)
            {
                dialogParam[i].pOutIsExit = &m_IsExit;
                dialogParam[i].pOutResult = &m_DialogResult;
                dialogParam[i].pOutExitCount = &m_ExitCount;
            }
            // OKボタン
            dialogParam[DialogButton::OK::Index].Result = DialogResult_OK;
            DialogButton[DialogButton::OK::Index].SetDefault();
            DialogButton[DialogButton::OK::Index].SetText("OK");
            DialogButton[DialogButton::OK::Index].SetParam((void*)&dialogParam[DialogButton::OK::Index]);
            DialogButton[DialogButton::OK::Index].OnOffMode(true);
            DialogButton[DialogButton::OK::Index].SetFunc(DialogFunc);
            // キャンセルボタン
            dialogParam[DialogButton::Cancel::Index].Result = DialogResult_Cancel;
            DialogButton[DialogButton::Cancel::Index].SetDefault();
            DialogButton[DialogButton::Cancel::Index].SetText("Cancel");
            DialogButton[DialogButton::Cancel::Index].SetParam((void*)&dialogParam[DialogButton::Cancel::Index]);
            DialogButton[DialogButton::Cancel::Index].OnOffMode(true);
            DialogButton[DialogButton::Cancel::Index].SetFunc(DialogFunc);
            // 承認ボタン
            dialogParam[DialogButton::Yes::Index].Result = DialogResult_Yes;
            DialogButton[DialogButton::Yes::Index].SetDefault();
            DialogButton[DialogButton::Yes::Index].SetText("Yes");
            DialogButton[DialogButton::Yes::Index].SetParam((void*)&dialogParam[DialogButton::Yes::Index]);
            DialogButton[DialogButton::Yes::Index].OnOffMode(true);
            DialogButton[DialogButton::Yes::Index].SetFunc(DialogFunc);
            // 否定ボタン
            dialogParam[DialogButton::NO::Index].Result = DialogResult_No;
            DialogButton[DialogButton::NO::Index].SetDefault();
            DialogButton[DialogButton::NO::Index].SetText("No");
            DialogButton[DialogButton::NO::Index].SetParam((void*)&dialogParam[DialogButton::NO::Index]);
            DialogButton[DialogButton::NO::Index].OnOffMode(true);
            DialogButton[DialogButton::NO::Index].SetFunc(DialogFunc);
            // その他のボタン
            dialogParam[DialogButton::Other::Index].Result = DialogResult_Ignore;
            DialogButton[DialogButton::Other::Index].SetDefault();
            DialogButton[DialogButton::Other::Index].SetText("");
            DialogButton[DialogButton::Other::Index].SetParam((void*)&dialogParam[DialogButton::Other::Index]);
            DialogButton[DialogButton::Other::Index].OnOffMode(true);
            DialogButton[DialogButton::Other::Index].SetFunc(DialogFunc);
        }
        m_DialogInfo = DialogInfo();
        m_DialogColor       = nn::util::Color4u8(nn::util::Color4u8::White().GetR(), nn::util::Color4u8::White().GetG(), nn::util::Color4u8::White().GetB(), 232);  // ダイアログのデフォルトカラー
        m_DialogShadowColor = nn::util::Color4u8(nn::util::Color4u8::Black().GetR(), nn::util::Color4u8::Black().GetG(), nn::util::Color4u8::Black().GetB(), 160);  // ダイアログのデフォルトの影の色

    }

    Dialog::~Dialog() NN_NOEXCEPT
    {

    }

    void Dialog::DialogFunc(void* pushButton, void* param)
    {
        NN_UNUSED(pushButton);
        Dialog::DialogParam* dialogParam = (Dialog::DialogParam*)param;
        NN_ASSERT_NOT_NULL(dialogParam);
        if (
            dialogParam->pOutExitCount != nullptr &&
            dialogParam->pOutIsExit != nullptr &&
            dialogParam->pOutResult != nullptr
            )
        {
            if (*dialogParam->pOutIsExit == false)
            {
                // 他の終了処理が実行されていない場合に処理を続行します
                *dialogParam->pOutResult = dialogParam->Result;         // 戻り値を変更します
                *dialogParam->pOutIsExit = true;                        // ダイアログの終了を許可します
                *dialogParam->pOutExitCount = 15;                       // 15 フレーム待って終了します
            }
        }
    }

    Dialog::DialogResult Dialog::ShowDialog() NN_NOEXCEPT
    {
        OpenDialogCount++;
        NN_ASSERT_LESS(OpenDialogCount, 4);
        //---------------------------------------------------------------------
        // 初期化処理
        //---------------------------------------------------------------------
        // 終了カウンタをリセットします
        m_ExitCount = 0;
        m_IsExit = false;

        // パフォーマンスメータが有効な場合、ダイアログの表示中も継続します
        bool isEnablePerfMeter = gGraphics.GetGraphicsSystem().IsEnablePerfMeter();

        // ダイアログを中央に表示させるための座標を計算する
        const nn::util::Float2 dialogPos    = NN_UTIL_FLOAT_2_INITIALIZER(
            (gGraphics.GetGraphicsSystem().GetScreenWidth() - m_DialogInfo.Size.x) / 2.f,
            (gGraphics.GetGraphicsSystem().GetScreenHeight() - m_DialogInfo.Size.y) / 2.f
        );
        // ダイアログのサイズ
        nn::util::Float2 dialogSize         = m_DialogInfo.Size;

        // ボタンに関する変数の初期化
        ButtonGroup dialogButtonGroup;                                                          // ダイアログが管理するボタン
        auto enableButtonCount        = m_DialogInfo.Buttons.CountPopulation();                       // 有効になっているボタンの個数
        auto buttonCount              = 0;                                                    // 実際に表示されているボタンの個数
        float buttonWidth = (m_DialogInfo.Size.x - (enableButtonCount + 1) * 16.f) / enableButtonCount; // ボタンの横幅

        // ダイアログのメインカラーと影の透過色を取得
        nn::util::Color4u8 dialogColorZero = m_DialogColor;
        dialogColorZero.SetA(0);
        nn::util::Color4u8 dialogShadowColorZero = m_DialogShadowColor;
        dialogShadowColorZero.SetA(0);

        // ボタンを作成します
        for (auto i = 0; i < m_DialogInfo.Buttons.GetCount() && buttonCount < enableButtonCount; ++i)
        {
            if (m_DialogInfo.Buttons.Test(i))
            {
                nns::hidfw::layout::Button button;
                auto itr = m_CustomButton.find(i);
                if (itr != m_CustomButton.end())
                {
                    button = itr->second;
                }
                else
                {
                    button = DialogButton[i];
                    button.SetPos(dialogPos.x + (i + 1) * 16.f + i * buttonWidth, dialogPos.y + m_DialogInfo.Size.y - 64.f);
                    button.SetSize(buttonWidth, 48.f);
                    button.SetMainColor(dialogColorZero);
                    button.SetBorderColor(button.GetEffectColor());
                    button.SetFunc(DialogFunc);
                    button.OnOffMode(true);
                }
                dialogButtonGroup.Add(new nns::hidfw::layout::Button(button));
                ++buttonCount;
            }
        }

        dialogButtonGroup.Focus(true);

        //---------------------------------------------------------------------
        // 前処理
        //---------------------------------------------------------------------
        /*
         *  4フレームかけて画面にぼかしをかけていきます
         */
        nns::hidfw::gfx::GraphicsDrawer::ScreenShotSlot slot = (nns::hidfw::gfx::GraphicsDrawer::ScreenShotSlot)(OpenDialogCount + 3);

        gDrawer.StartScanScreen();
        for (auto i = 0; i < 4; ++i)
        {
            gDrawer.BeginDraw();
            {
                gDrawer.EnableBlurhader(true);  // ぼかしシェーダを有効にする
                gDrawer.Draw2DPrevScreen(nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(1280.f, 720.f));
                gDrawer.EnableBlurhader(false);
                gDrawer.ScreenShot(slot);
            }
            gDrawer.EndDraw();
            if (isEnablePerfMeter)
            {
                gGraphics.GetGraphicsSystem().EndPerfFrame();
                gGraphics.GetGraphicsSystem().BeginPerfFrame();
            }
        }
        //gDrawer.StopScanScreen();

        //---------------------------------------------------------------------
        // メインループ
        //---------------------------------------------------------------------
        uint64_t frameCount = 0;
        do
        {
            gInput.Update();

            // 接続中のコントローラのボタンの押下げ状況を確認します
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                if (
                    ((*it)->IsTrigger(nn::hid::NpadButton::Plus::Mask) || (*it)->IsTrigger(nn::hid::NpadButton::Minus::Mask)) &&
                    (m_DialogInfo.Buttons.Test<DialogButton::Cancel>() || m_DialogInfo.Buttons.IsAllOff())
                    )
                {
                    /*
                     *  キャンセルボタンが配置してある場合か、ボタンが一つも配置されていない場合は
                     *  (+) もしくは (-) ボタンが押下げられた場合、ダイアログを即時キャンセルします
                     */
                    m_DialogResult = DialogResult_Cancel;
                    m_ExitCount = 0;
                    m_IsExit = true;
                }
            }

            dialogButtonGroup.Update();

            for (int i = 0; i < dialogButtonGroup.GetListSize(); ++i)
            {
                if (dialogButtonGroup.GetItemSet().at(i)->IsSelected())
                {
                    // いずれかのボタンが押された場合、フォーカスを固定し以降の動作指定を無視する
                    dialogButtonGroup.Focus(true);
                    break;
                }
            }

            if (m_DialogInfo.UpdateFunc != nullptr)
            {
                m_DialogInfo.UpdateFunc(this, m_DialogInfo.UpdateFuncParam);
            }

            gDrawer.BeginDraw();            // 描画関数の受付開始
            {
                // 背景
                gDrawer.SetColor(nn::util::Color4u8::White());
                gDrawer.Draw2DScreenShot(slot, nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(1280.f, 720.f));
                // 背景を暗くする
                gDrawer.SetColor(nn::util::Color4u8(0, 2, 4, 204));
                gDrawer.Draw2DRect(nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(1280.f, 720.f));
                // ダイアログの影
                gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_DialogShadowColor, dialogShadowColorZero);
                gDrawer.Draw2DRoundedFrame(dialogPos, dialogSize, 0.05f, 48, 12.f);
                // ダイアログの外枠
                gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_DialogColor, dialogColorZero);
                gDrawer.Draw2DRoundedFrame(dialogPos, dialogSize, 0.05f, 48, 1.5f);
                // ダイアログ本体
                gDrawer.SetColor(m_DialogColor);
                gDrawer.Draw2DRoundedRect(dialogPos, dialogSize, 0.05f, 48);
                // タイトルとメッセージの分離線
                gDrawer.SetColor(nn::util::Color4u8::Lerp(nn::util::Color4u8::Gray(), m_DialogColor, 0.5f));
                gDrawer.Draw2DLine(
                    nn::util::MakeFloat2(dialogPos.x - 1.f, dialogPos.y + 64.f),
                    nn::util::MakeFloat2(dialogPos.x + dialogSize.x + 1.f, dialogPos.y + 64.f));
                // ボタンを表示します
                dialogButtonGroup.Draw();

                if (m_DialogInfo.DrawerFunc != nullptr)
                {
                    m_DialogInfo.DrawerFunc(this, m_DialogInfo.DrawFuncParam);
                }
            }
            gDrawer.EndDraw();

            if (isEnablePerfMeter)
            {
                gGraphics.GetGraphicsSystem().EndPerfFrame();
                gGraphics.GetGraphicsSystem().BeginPerfFrame();
            }

            ++frameCount;

            if (m_ExitCount != 0) { --m_ExitCount; }

        } while ((m_ExitCount > 0) || (!m_IsExit));

        OpenDialogCount--;
        NN_ASSERT_GREATER_EQUAL(OpenDialogCount, 0);

        // 元の画面に戻る前にコントローラの状態を更新します
        gInput.Update();

        gDrawer.StartScanScreen();

        return m_DialogResult;
    } // NOLINT(impl/function_size)

    Dialog::DialogResult Dialog::ShowDialog(const DialogInfo& info) NN_NOEXCEPT
    {
        SetDialogInfo(info);
        return ShowDialog();
    }

    void Dialog::CloseDialog() NN_NOEXCEPT
    {
        m_DialogResult = DialogResult_Cancel;
        m_ExitCount = 0;
        m_IsExit = true;
    }

    void Dialog::SetDialogButton(int index, const nns::hidfw::layout::Button& button) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(index, 0);
        NN_ASSERT_LESS(index, 8);

        auto itr = m_CustomButton.find(index);

        if (itr != m_CustomButton.end())
        {
            m_CustomButton.erase(itr);
        }

        m_CustomButton.insert(std::make_pair(index, button));
    }

    void Dialog::ClearDialogButton() NN_NOEXCEPT
    {
        m_CustomButton.erase(m_CustomButton.begin(), m_CustomButton.end());
    }

}}}
