﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "layout_BarChart.h"

namespace nns { namespace hidfw { namespace layout {

    BarChart::BarChart() NN_NOEXCEPT
    {
        SetDefault();
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_EffectCount); ++i)
        {
            m_EffectCount[i] = 0;
        }
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_EffectValues); ++i)
        {
            m_EffectValues[i] = 0.f;
        }
    }

    void BarChart::SetDefault() NN_NOEXCEPT
    {
        static_cast<BaseItem*>(this)->SetDefault();
        SetSize(600.f, 300.f);
        m_DivisionCount = 10;
        m_ElementsCount = 0;

        m_MainColor     = nn::util::Color4u8::Black();  // 背景色
        m_SubColor      = nn::util::Color4u8::White();  // 棒線の色
        m_TextColor     = nn::util::Color4u8::Blue();   // 未使用
        m_BorderColor   = nn::util::Color4u8::Gray();   // 枠線の色
        m_EffectColor   = nn::util::Color4u8::Green();  // ホバー時に表示されるエフェクト色

        SetViewerMode(ViewerMode_Default);
        SetMaxVerticalValue(1.f);
        SetMinVerticalValue(0.f);
        SetMaxHorizontalValue(1.f);
        SetMinHorizontalValue(0.f);
    }

    void BarChart::Update() NN_NOEXCEPT
    {
        for (
            std::vector<Element>::iterator it = m_Elements.begin();
            it != m_Elements.end();
            ++it
            )
        {
            float verticalScale =
                ((*it).VerticalValue - m_MinVerticalValue) / m_MaxVerticalValue;

            (*it).DivisionNumber = ((((*it).HorizontalValue - m_MinHorizontalValue) / (m_MaxHorizontalValue - m_MinHorizontalValue)) * static_cast<float>(m_DivisionCount));

            int32_t divIndex = static_cast<int>((*it).DivisionNumber + 0.5f);
            if ((divIndex > 0) && (static_cast<size_t>(divIndex) < MaxDivisionCount))
            {
                if (m_EffectValues[divIndex] < verticalScale * m_Size.y)
                {
                    m_EffectValues[divIndex] = verticalScale * m_Size.y;
                    m_EffectCount[divIndex] = 0;
                }
            }
        }

        switch (m_ViewerMode)
        {
        case nns::hidfw::layout::BarChart::ViewerMode_Default:
        {
            break;
        }
        case nns::hidfw::layout::BarChart::ViewerMode_Music:
        {
            for (int i = 0; i < MaxDivisionCount; ++i)
            {
                if (i >= m_DivisionCount || m_EffectValues[i] < 0.001f)
                {
                    m_EffectValues[i] = 0.f;
                    m_EffectCount[i] = 0;
                }
                else
                {
                    if (m_EffectCount[i] > 15 && m_EffectValues[i] > 0.001f)
                    {
                        m_EffectValues[i] -= static_cast<float>((m_EffectCount[i] - 15) * (m_EffectCount[i]) - 15) / 600.f;
                        if (m_EffectValues[i] < 0.f)
                        {
                            m_EffectValues[i] = 0.f;
                        }
                    }
                    ++m_EffectCount[i];
                }
            }
            m_OldElements.clear();
            m_OldElements = m_Elements;
            m_ElementsCount = m_Elements.size();
            Clear();
            break;
        }
        default : NN_UNEXPECTED_DEFAULT;
        }
        if (m_State.Test<LayoutState::Selected>())
        {
            CallFunc();
            Cancel();
        }
        ++m_FrameCount;
    }

    void BarChart::PrintText() NN_NOEXCEPT
    {

    }

    void BarChart::Draw() NN_NOEXCEPT
    {
        nn::util::Color4u8 frameColor     = m_BorderColor;    // 枠線の色 (異常時には色を変更します)
        nn::util::Color4u8 subBorderColor = m_BorderColor;    // 中央の横線の色
        subBorderColor.SetA(subBorderColor.GetA() / 2);

        if (IsVisible())
        {
            //　描画が有効な場合
            //----------------------------------------------------------------
            //  背景と補助線の描画
            //----------------------------------------------------------------
            // 背景
            gDrawer.SetColor(m_MainColor);
            gDrawer.Draw2DRect(m_Pos, m_Size);

            // 横
            gDrawer.SetColor(subBorderColor);
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + m_Size.y / 4.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + m_Size.y / 4.f)
            );
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + (m_Size.y / 4.f) * 3.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + (m_Size.y / 4.f) * 3.f)
            );
            gDrawer.SetColor(m_BorderColor);
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(m_Pos.x, m_Pos.y + m_Size.y / 2.f),
                nn::util::MakeFloat2(m_Pos.x + m_Size.x, m_Pos.y + m_Size.y / 2.f)
            );

            //----------------------------------------------------------------
            //  要素毎の描画処理
            //----------------------------------------------------------------
            // 描画される値の横幅
            float elementWidth = m_Size.x / static_cast<float>((m_DivisionCount < 1) ? 1 : m_DivisionCount);
            // ViewerMode_Music の場合 m_OldElements　を使用します
            std::vector<Element>& elements = (m_ViewerMode == ViewerMode_Music) ? m_OldElements : m_Elements;
            // 値の描画
            for (
                std::vector<Element>::iterator it = elements.begin();
                it != elements.end();
                ++it
                )
            {
                float verticalScale = ((*it).VerticalValue - m_MinVerticalValue) / m_MaxVerticalValue;          // 縦幅のスケール
                float elementPos    = m_Pos.x + elementWidth * static_cast<int>((*it).DivisionNumber + 0.5f);   // 横軸の描画座標

                if ((elementPos < m_Pos.x) || elementPos > (m_Pos.x + m_Size.x))
                {
                    // 横方向の値が上限・加減を超過した場合
                    //frameColor = nn::util::Color4u8::Red();
                }
                else
                {
                    gDrawer.SetColor(
                        (verticalScale < 0.0001f) ? nn::util::Color4u8::Blue() :
                        (verticalScale > 1.0001f) ? nn::util::Color4u8::Red() :
                        m_SubColor);
                    gDrawer.Draw2DRect(
                        nn::util::MakeFloat2(elementPos, m_Pos.y + (m_Size.y - verticalScale * m_Size.y)),
                        nn::util::MakeFloat2(elementWidth, verticalScale * m_Size.y)
                    );
                }
            }

            gDrawer.SetColor(m_SubColor);

            switch (m_ViewerMode)
            {
            case nns::hidfw::layout::BarChart::ViewerMode_Default:
            {
                break;
            }
            case nns::hidfw::layout::BarChart::ViewerMode_Music:
            {
                for (int i = 0; i < MaxDivisionCount; ++i)
                {
                    if ((i < m_DivisionCount) && (m_EffectValues[i] > 0.0001f))
                    {
                        float elementPos = m_Pos.x + elementWidth * static_cast<float>(i);
                        gDrawer.Draw2DLine(
                            nn::util::MakeFloat2(elementPos, m_Pos.y + (m_Size.y - m_EffectValues[i])),
                            nn::util::MakeFloat2(elementPos + elementWidth, m_Pos.y + (m_Size.y - m_EffectValues[i]))
                        );
                    }
                }
                break;
            }
            default : NN_UNEXPECTED_DEFAULT;
            }
        }

        if (IsVisible() || m_IsFrameDraw)
        {
            // 表示が有効か、外枠のみ表示設定になっている場合
            if (m_State.Test<LayoutState::Hover>())
            {
                // ホバー時のエフェクトの色
                const nn::util::Color4u8 transparenceEffectColor = nn::util::Color4u8(m_EffectColor.v[0], m_EffectColor.v[1], m_EffectColor.v[2], 0);
                float effect = nn::util::SinEst(nn::util::DegreeToRadian((m_FrameCount % 120) * 3)) * 1.f + 5.f;
                gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_EffectColor, transparenceEffectColor);
                gDrawer.Draw2DFrame(m_Pos, m_Size, effect);
            }
            gDrawer.SetColor(frameColor);
            gDrawer.Draw2DFrame(m_Pos, m_Size, 1);
        }
    }

    void BarChart::AddValue(float horizontal, float vertical) NN_NOEXCEPT
    {
        Element element = Element(horizontal, vertical);
        m_Elements.push_back(element);
    }

    void BarChart::Clear() NN_NOEXCEPT
    {
        m_Elements.clear();
    }


}}}
