﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hid_Vibration.h"

namespace nns { namespace hidfw { namespace hid {

    void Vibration::Initialize(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        m_IsEnable  = true;
        m_NpadId    = npadId;

        m_DeviceCount[nn::hid::NpadStyleFullKey::Index]     = nn::hid::GetVibrationDeviceHandles(m_Handle[nn::hid::NpadStyleFullKey::Index], DeviceCountMax, m_NpadId, nn::hid::NpadStyleFullKey::Mask);
        m_DeviceCount[nn::hid::NpadStyleHandheld::Index]    = nn::hid::GetVibrationDeviceHandles(m_Handle[nn::hid::NpadStyleHandheld::Index], DeviceCountMax, m_NpadId, nn::hid::NpadStyleHandheld::Mask);
        m_DeviceCount[nn::hid::NpadStyleJoyDual::Index]     = nn::hid::GetVibrationDeviceHandles(m_Handle[nn::hid::NpadStyleJoyDual::Index], DeviceCountMax, m_NpadId, nn::hid::NpadStyleJoyDual::Mask);
        m_DeviceCount[nn::hid::NpadStyleJoyLeft::Index]     = nn::hid::GetVibrationDeviceHandles(m_Handle[nn::hid::NpadStyleJoyLeft::Index], DeviceCountMax, m_NpadId, nn::hid::NpadStyleJoyLeft::Mask);
        m_DeviceCount[nn::hid::NpadStyleJoyRight::Index]    = nn::hid::GetVibrationDeviceHandles(m_Handle[nn::hid::NpadStyleJoyRight::Index], DeviceCountMax, m_NpadId, nn::hid::NpadStyleJoyRight::Mask);

        for (size_t styleIndex = 0; styleIndex < StyleCountMax; ++styleIndex)
        {
            for (size_t handleIndex = 0; handleIndex < static_cast<size_t>(m_DeviceCount[styleIndex]); ++handleIndex)
            {
                nn::hid::InitializeVibrationDevice(m_Handle[styleIndex][handleIndex]);
                nn::hid::GetVibrationDeviceInfo(&m_VibrationInfo[styleIndex][handleIndex], m_Handle[styleIndex][handleIndex]);
            }
        }
        for (size_t i = 0; i < DeviceCountMax; ++i)
        {
            m_VibrationValue[i] = nn::hid::VibrationValue::Make();
            m_CurrentVibrationValue[i] = nn::hid::VibrationValue::Make();
            m_VibrationCount[i] = 0;
        }
    }

    void Vibration::Update() NN_NOEXCEPT
    {
        m_OldNpadStyleSet = m_NpadStyleSet;
        m_NpadStyleSet.Reset();
        m_NpadStyleSet = nn::hid::GetNpadStyleSet(m_NpadId);

        for (size_t handleIndex = 0; handleIndex < DeviceCountMax; ++handleIndex)
        {
            // 現在の振動状態をクリアします
            m_CurrentVibrationValue[handleIndex] = nn::hid::VibrationValue::Make();
        }

        for (size_t styleIndex = 0; styleIndex < StyleCountMax; ++styleIndex)
        {
            for (size_t handleIndex = 0; handleIndex < static_cast<size_t>(m_DeviceCount[styleIndex]); ++handleIndex)
            {
                if (m_NpadStyleSet.Test(styleIndex))
                {
                    // 有効なスタイルが存在する場合、有効なスタイルのハンドルにのみ振動を送信します
                    // 振動を送信します
                    if (m_IsEnable)
                    {
                        nn::hid::SendVibrationValue(m_Handle[styleIndex][handleIndex], m_VibrationValue[handleIndex]);
                    }
                    // 現在の振動値を取得します
                    nn::hid::GetActualVibrationValue(&m_CurrentVibrationValue[handleIndex], m_Handle[styleIndex][handleIndex]);

                    m_OldVibrationValue[handleIndex] = m_VibrationValue[handleIndex];
                }
                else if(m_OldNpadStyleSet.Test(styleIndex))
                {
                    // スタイルが変更された場合は元のスタイルの振動は停止します
                    nn::hid::SendVibrationValue(m_Handle[styleIndex][handleIndex], nn::hid::VibrationValue::Make());
                }
                else
                {
                    // 何もしない
                }
            }
        }

        for (size_t handleIndex = 0; handleIndex < DeviceCountMax; ++handleIndex)
        {
            // 振動をリセットします
            m_VibrationValue[handleIndex] = nn::hid::VibrationValue::Make();
        }
    }

    void Vibration::Send(const nn::hid::VibrationValue& vib) NN_NOEXCEPT
    {
        for (size_t i = 0; i < DeviceCountMax; ++i)
        {
            m_VibrationValue[i] = vib;
        }
    }

    void Vibration::Send(const nn::hid::VibrationValue& leftVib, const nn::hid::VibrationValue& rightVib) NN_NOEXCEPT
    {
        m_VibrationValue[0] = leftVib;
        m_VibrationValue[1] = rightVib;
    }

    void Vibration::SendLeft(const nn::hid::VibrationValue& leftVib) NN_NOEXCEPT
    {
        m_VibrationValue[0] = leftVib;
    }

    void Vibration::SendRight(const nn::hid::VibrationValue& rightVib) NN_NOEXCEPT
    {
        m_VibrationValue[1] = rightVib;
    }

    void Vibration::StopVibration() NN_NOEXCEPT
    {
        Send(nn::hid::VibrationValue::Make());
    }

    void Vibration::StopLeftVibration() NN_NOEXCEPT
    {
        SendLeft(nn::hid::VibrationValue::Make());
    }

    void Vibration::StopRightVibration() NN_NOEXCEPT
    {
        SendRight(nn::hid::VibrationValue::Make());
    }

    void Vibration::GetCurVibrationValue(nn::hid::VibrationValue* leftVib, nn::hid::VibrationValue* rightVib) NN_NOEXCEPT
    {
        if (leftVib != nullptr)
        {
            *leftVib = m_CurrentVibrationValue[0];
        }
        if (rightVib != nullptr)
        {
            *rightVib = m_CurrentVibrationValue[1];
        }
    }

    int Vibration::GetVibrationDeviceHandle(nn::hid::NpadStyleSet style, nn::hid::VibrationDeviceHandle* handle, int count) NN_NOEXCEPT
    {
        NN_ASSERT(handle);
        int getHandleCount = 0;

        if (style.IsAnyOn())
        {
            int styleIndex = 0;
            for (; styleIndex < nns::hidfw::hid::Vibration::StyleCountMax; ++styleIndex)
            {
                if (style.Test(styleIndex))
                {
                    break;
                }
            }
            if (styleIndex < nns::hidfw::hid::Vibration::StyleCountMax)
            {
                for (; (getHandleCount < count) && (getHandleCount <m_DeviceCount[styleIndex]); ++getHandleCount)
                {
                    handle[getHandleCount] = m_Handle[styleIndex][getHandleCount];
                }
            }
        }
        return getHandleCount;
    }

    int Vibration::GetCurSendVibrationValue(nn::hid::VibrationValue* vib, int count) NN_NOEXCEPT
    {
        NN_ASSERT(vib);
        int getValueCount = 0;

        for (; (getValueCount < count) && (getValueCount < DeviceCountMax); ++getValueCount)
        {
            vib[getValueCount] = m_OldVibrationValue[getValueCount];
        }

        return getValueCount;
    }

    void Vibration::Enable() NN_NOEXCEPT
    {
        m_IsEnable = true;
    }

    void Vibration::Disable() NN_NOEXCEPT
    {
        m_IsEnable = false;
    }

    bool Vibration::IsEnable() NN_NOEXCEPT
    {
        return m_IsEnable;
    }

}}}
