﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラ操作の管理のための API の宣言
 */

#pragma once

#include <vector>

#include "hid_Controller.h"

namespace nns { namespace hidfw { namespace hid {

    /**
     * @brief       複数コントローラ関連の設定、状態取得等を統括する為のクラスです
     * @details     複数台のコントローラを、Controllerクラスを用いて統括処理します
     */
    class ControllerManager
    {
        NN_DISALLOW_COPY(ControllerManager);
        NN_DISALLOW_MOVE(ControllerManager);

    public:
        //----------------------------------------------------------------
        //! @brief              コントローラマネージャの初期化
        //! @details            コントローラををリストに追加します
        //!                     コントローラ自体の初期化処理は本関数では行いません
        //----------------------------------------------------------------
        void Initialize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラマネージャの破棄
        //! @details            コントローラリストの内容を破棄します
        //----------------------------------------------------------------
        void Finalize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラの更新
        //! @details            コントローラリストに追加されてある
        //!                     全てのコントローラの Update() を実行します
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラの更新
        //! @details            コントローラリストに追加されてある
        //!                     全てのコントローラの Update() を実行します
        //----------------------------------------------------------------
        static ControllerManager& GetInstance() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラリストのサイズ
        //! @return             コントローラリストのサイズを返します
        //----------------------------------------------------------------
        int GetControllerListSize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              接続中のコントローラ数
        //! @return             接続中のコントローラが数を返します
        //----------------------------------------------------------------
        int GetConnectedControllerListSize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラリストを取得します
        //! @details            NpadId No1～No8 及び　Handheld のコントローラのリストを取得します
        //!                     NpadId が有効か否かは考慮しません
        //! @return             コントローラのリストを返します
        //----------------------------------------------------------------
        std::vector<nns::hidfw::hid::Controller*>& GetControllerList() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラの割り当てを変更します
        //! @details            現在接続状態のコントローラのリストを返します
        //! @return             接続中のコントローラのリストを返します
        //----------------------------------------------------------------
        std::vector<nns::hidfw::hid::Controller*>& GetConnectedControllerList() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              指定したインデックスに設定されているコントローラを返します
        //! @details            存在しないコントローラのインデックスを指定した場合 nullptr が返ります
        //! @pre                index < GetControllerList()
        //! @param[in] index    コントローラのインデックス
        //! @return             コントローラを返します
        //----------------------------------------------------------------
        Controller* GetControllerFromIndex(int index) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              指定したNpadIdのコントローラを返します
        //! @details            不正なNpadIdを指定した場合 nullptr が返ります
        //! @param[in] npadId   NpadId
        //! @return             コントローラを返します
        //----------------------------------------------------------------
        Controller* GetController(nn::hid::NpadIdType npadId) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              指定したインデックスに設定されているコントローラを返します
        //! @details            存在しないコントローラのインデックスを指定した場合 nullptr が返ります
        //! @pre                index < GetConnectedControllerList()
        //! @param[in] index    コントローラのインデックス
        //! @return             接続中のコントローラを返します
        //----------------------------------------------------------------
        Controller* GetConnectedControllerFromIndex(int index) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              接続中のコントローラのうち指定したNpadIdのコントローラを返します
        //! @details            不正なNpadIdを指定した場合 nullptr が返ります
        //! @param[in] npadId   NpadId
        //! @return             接続中のコントローラを返します
        //----------------------------------------------------------------
        Controller* GetConnectedController(nn::hid::NpadIdType npadId) NN_NOEXCEPT;

    private:
        //----------------------------------------------------------------
        //! @brief GetInstance の内部で呼び出されます
        //----------------------------------------------------------------
        ControllerManager() NN_NOEXCEPT;

    private:
        std::vector<nns::hidfw::hid::Controller*> m_pControllerList;               //!< 全てのコントローラのリスト
        std::vector<nns::hidfw::hid::Controller*> m_pConnectedControllerList;;     //!< 接続中のコントローラのリスト
    };

}}}
