﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラ操作(ボタン、スティック)とタッチパネル操作を総合的に管理するAPIの宣言
 * @details     本ヘッダのインクルードで、コントローラ操作とタッチパネル操作が全て可能になります
 */

#pragma once

#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include "hid/hid_ControllerManager.h"
#include "hid/hid_Touch.h"

namespace nns { namespace hidfw { namespace hid {

    /**
     * @brief       コントローラ操作とタッチパネル操作関連処理を総合的に管理するクラスです
     * @details     操作に関するロギング処理の管理も、本クラスで実施します
     */
    class Input
    {
        NN_DISALLOW_COPY(Input);
        NN_DISALLOW_MOVE(Input);
    public:
        enum KeyLoggerOutputMode
        {
            KeyLoggerOutputMode_Sd = 0,                     //!< SDカードに出力
            KeyLoggerOutputMode_Host = 1,                   //!< HOST接続されているPCに出力
            KeyLoggerOutputMode_TargetManager,              //!< TargetManagerに出力
        };
        /**
         * @brief       ロギングを行う機能を指定する為の構造体です
         */
        struct KeyLoggerTarget
        {
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<0> Style;                     //!< Npadのスタイルを出力対象にします
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<1> ButtonSet;                 //!< 押下げられているボタンを出力対象にします
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<2> AnalogStick;               //!< アナログスティックを出力対象にします
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<3> SixAxisSensor;             //!< 六軸センサを出力対象にします
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<4> HoldType;                  //!< 持ち方を出力対象にします
            typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget>::Flag<5> HandheldActivationMode;    //!< 本体装着コンが有効になる本数を出力対象にします
        };
        typedef ::nn::util::BitFlagSet<32, KeyLoggerTarget> KeyLoggerTargetet;

    public:
        //----------------------------------------------------------------
        //! @brief インスタンスの取得
        //----------------------------------------------------------------
        static Input& GetInstance() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      コントローラとタッチパネルの初期化処理
        //! @details    全てのNpadId (No1～No8 及び Handheld) と
        //!             全てのNpadStyle が有効化されます
        //----------------------------------------------------------------
        void Initialize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      タッチパネルとコントローラの更新処理
        //! @details    毎フレーム先頭付近で呼び出して下さい
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 終了処理
        //----------------------------------------------------------------
        void Finalize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                      ログの出力対象を指定します
        //! @param[in] loggingTarget    出力対象とするデータを指定します
        //----------------------------------------------------------------
        void SetLoggingTarget(const KeyLoggerTargetet& loggingTarget) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      (デバッグ用) キー入力の記録を開始します
        //! @details
        //! @param[in] outputMode   出力方式を設定します
        //!                         StartKeyLoggerHost と同時使用は出来ません
        //!                         既に開始済みの場合は何もせずに処理を終了します
        //!                         KeyLoggerOutputMode_SD : ログをSDカードに出力します
        //!                         KeyLoggerOutputMode_HOST : StartKeyLoggerHost を使用してください
        //!                         KeyLoggerOutputMode_TargetManager : ログをTargetManager上に出力します
        //! @pre        KeyLoggerOutputMode_SD : SDカードが挿入されていること
        //! @pre        KeyLoggerOutputMode_TargetManager : TargetManagerと接続が完了していること
        //----------------------------------------------------------------
        bool StartKeyLogger(KeyLoggerOutputMode outputMode) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      (デバッグ用) キー入力の記録を開始します
        //! @details    outFilePath で指定したパスの直下に Log ディレクトリを作成し
        //!             ログファイルを出力します
        //!             StartKeyLogger と同時使用は出来ません
        //!             既に開始済みの場合は何もせずに処理を終了します
        //! @param[in]  outFilePath  ログを出力するHOST側のファイルパス
        //! @pre        outFilePath がアクセス・書き込みが可能なパスであること
        //----------------------------------------------------------------
        bool StartKeyLoggerHost(const char* outFilePath) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief キー入力の記録を停止します
        //----------------------------------------------------------------
        void StopKeyLogger() NN_NOEXCEPT;

    private:
        //----------------------------------------------------------------
        //! @brief GetInstance の内部で呼び出されます
        //----------------------------------------------------------------
        Input() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief (デバッグ用) 入力ログを書き込みます
        //! @pre   StartKeyLogger が呼び出されている
        //----------------------------------------------------------------
        void WriteLog() NN_NOEXCEPT;

    private:
        bool                    m_IsStartedKeyLogger;
        nn::fs::FileHandle      m_KeyLoggerFileHandle;
        KeyLoggerOutputMode     m_OutputMode;
        KeyLoggerTargetet       m_OutputTarget;
    };

}}}

#define gInput (nns::hidfw::hid::Input::GetInstance())
#define gController (nns::hidfw::hid::ControllerManager::GetInstance())
#define gTouch (nns::hidfw::hid::Touch::GetInstance())
