﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <locale>
#include <codecvt>
#include <unordered_map>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include <nn/pl.h>
#include <nn/util/util_CharacterEncoding.h>

#include "../gfx.h"

namespace nns { namespace hidfw { namespace gfx {

    struct FontData
    {
        uint32_t code;
        uint16_t xPos;
        uint16_t yPos;
        nn::fontll::GlyphMap glyp;
    };

    struct LineData
    {
        uint16_t lineSize;
        uint16_t nextWritePos;
    };

    // ハンドル値
    //   未定義      size   ----  outline  ----
    // XXXX XXXX  0000 0000 0000 0000 0000 0000
    //               font code
    // 0000 0000 0000 0000 0000 0000 0000 0000
    typedef uint64_t FontHandle;

    /**
    * @brief       内蔵フォントを利用するためのクラスです
    * @details
                実際のフォントの描画には GraphicsDrawer の Print と DrawText をご利用下さい
                本クラスでは不要になったフォントデータの削除は行っていません
                英数字のみを利用する場合はキャッシュサイズに十分の余裕がありますが
                多言語対応する場合はキャッシュが不足し文字描画に失敗する可能性があります

    */
    class FontSystem
    {
        //NN_DISALLOW_COPY(FontSystem);
        //NN_DISALLOW_MOVE(FontSystem);
    public:
        static const size_t                         MemoryPoolSize = 10 * 1024 * 1024;
        static const int                            TextureHeight = 4096;
        static const int                            TextureWidth = 4096;
        static const int                            TextureSize = TextureHeight * TextureWidth;
        static const int                            PrintTextCountMax = 1024;
        static const int                            FontSizeMin = 16;
        static const int                            FontSizeMax = 128;
    public:
        FontSystem();

        static FontSystem& GetInstance() NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;

        void SetPos(float x, float y) NN_NOEXCEPT;

        void SetPosX(float x) NN_NOEXCEPT;

        void SetPosY(float y) NN_NOEXCEPT;

        nn::util::Float2 GetPos() const NN_NOEXCEPT;

        float GetPosX() const NN_NOEXCEPT;

        float GetPosY() const NN_NOEXCEPT;

        void SetSize(const uint32_t fontSize) NN_NOEXCEPT;

        int SetFlags(const nn::fontll::ScalableFontEngine::Flags flag) NN_NOEXCEPT;

        int SetBorderWidth(const uint16_t width) NN_NOEXCEPT;

        FontHandle CreateFont(const uint32_t charCode) NN_NOEXCEPT;

        bool CheckCacheFont(FontHandle* pOutHandle, const uint32_t charCode) NN_NOEXCEPT;

        void PrintChar(const uint32_t ch) NN_NOEXCEPT;

        int Print(const char* str, ...) NN_NOEXCEPT;

        void Flush() NN_NOEXCEPT;

        const uint32_t* GetIndexBuffer() const NN_NOEXCEPT;
        const nn::util::Float3* GetVertexBuffer() const NN_NOEXCEPT;
        const nn::util::Float2* GetUvBuffer() const NN_NOEXCEPT;
        const nn::util::Float4* GetColorBuffer() const NN_NOEXCEPT;

        size_t GetIndexBufferSize() const NN_NOEXCEPT;
        size_t GetVertexBufferSize() const NN_NOEXCEPT;
        size_t GetUvBufferSize() const NN_NOEXCEPT;
        size_t GetColorBufferSize() const NN_NOEXCEPT;

        int GetIndexBufferCount() const NN_NOEXCEPT;
        int GetVertexBufferCount() const NN_NOEXCEPT;
        int GetUvBufferCount() const NN_NOEXCEPT;
        int GetColorBufferCount() const NN_NOEXCEPT;

        nn::gfx::DescriptorSlot* GetDescriptorSlot() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画領域を指定してテキストの描画を行います
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void DrawText() NN_NOEXCEPT;

        void DrawFontTexture() NN_NOEXCEPT;

    private:
        nn::mem::StandardAllocator                  m_ApplicationHeap;
        int                                         m_PrintTextCount;
        uint32_t*                                   m_pIndexBuffer;
        nn::util::Float3*                           m_pVertexBuffer;
        nn::util::Float2*                           m_pUvBuffer;
        nn::util::Float4*                           m_pColorBuffer;
        nn::fontll::ScalableFontEngine              m_StandardFont;             // 複数のフォントを読み込んでも1つしか利用が出来なかったため、フォント舞に作成
        nn::fontll::ScalableFontEngine              m_NintendoExtensionFont;
        char                                        m_StandardFontName[nn::fontll::FontNameLengthMax];
        char                                        m_NintendoExtensionFontName[nn::fontll::FontNameLengthMax];
        uint32_t                                    m_FontSize;
        float                                       m_AscentRatio;
        std::unordered_map<FontHandle, FontData>    m_FontData;
        std::vector<LineData>                       m_LineData;
        nn::fontll::Metrics                         m_Metrics;
        int                                         m_OutLineWidth;
        nn::gfx::Texture                            m_FontTexture;
        nn::gfx::Buffer                             m_FontTextureBuffer;
        nn::gfx::TextureView                        m_FontTextureView;
        nn::gfx::DescriptorSlot                     m_FontTextureSlot;
        nn::util::Float2                            m_PrintPosition;
        int                                         m_BufferOffset;
    };
}}}

#define gFont (nns::hidfw::gfx::FontSystem::GetInstance())
