﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfx_FontSystem.h"

namespace nns { namespace hidfw { namespace gfx {

    FontSystem::FontSystem() NN_NOEXCEPT { m_PrintTextCount = 0; }

    FontSystem& FontSystem::GetInstance() NN_NOEXCEPT
    {
        static FontSystem instance;
        return instance;
    }

    void FontSystem::Initialize() NN_NOEXCEPT
    {
        setlocale(LC_CTYPE, "jpn");

        // 標準フォントの読み込みを行います
        nn::pl::RequestSharedFontLoad(nn::pl::SharedFontType_Standard);
        while (nn::pl::SharedFontLoadState_Loading == nn::pl::GetSharedFontLoadState(nn::pl::SharedFontType_Standard))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
        // フォントに利用するメモリを確保します
        m_pIndexBuffer = nullptr;
        m_pVertexBuffer = nullptr;
        m_pUvBuffer = nullptr;
        m_pColorBuffer = nullptr;
        m_ApplicationHeap.Initialize((void*)malloc(128 * 1024 * 1024), 128 * 1024 * 1024);
        m_pIndexBuffer = (uint32_t*)m_ApplicationHeap.Allocate(sizeof(uint32_t) * PrintTextCountMax * 6, sizeof(uint32_t));
        NN_ASSERT_NOT_NULL(m_pIndexBuffer);
        m_pVertexBuffer = (nn::util::Float3*)m_ApplicationHeap.Allocate(sizeof(nn::util::Float3) * PrintTextCountMax * 4, sizeof(float));
        NN_ASSERT_NOT_NULL(m_pVertexBuffer);
        m_pUvBuffer = (nn::util::Float2*)m_ApplicationHeap.Allocate(sizeof(nn::util::Float2) * PrintTextCountMax * 4, sizeof(float));
        NN_ASSERT_NOT_NULL(m_pUvBuffer);
        m_pColorBuffer = (nn::util::Float4*)m_ApplicationHeap.Allocate(sizeof(nn::util::Float4) * PrintTextCountMax * 4, sizeof(float));
        NN_ASSERT_NOT_NULL(m_pColorBuffer);
        // フォントの保存用テクスチャを作成します
        nn::gfx::TextureInfo textureInfo;
        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
        textureInfo.SetWidth(TextureWidth);
        textureInfo.SetHeight(TextureHeight);
        nns::hidfw::gfx::Graphics::GetInstance().GetGraphicsSystem().AllocateTexture(&m_FontTexture, &textureInfo);
        // フォントのコピー用バッファを作成します
        nn::gfx::BufferInfo bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        bufferInfo.SetSize(TextureWidth * TextureHeight);
        nns::hidfw::gfx::Graphics::GetInstance().GetGraphicsSystem().AllocateBuffer(&m_FontTextureBuffer, &bufferInfo);
        // ビューの作成
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
        info.SetChannelMapping(
            nn::gfx::ChannelMapping_One,
            nn::gfx::ChannelMapping_One,
            nn::gfx::ChannelMapping_One,
            nn::gfx::ChannelMapping_Red);
        info.SetTexturePtr(&m_FontTexture);
        m_FontTextureView.Initialize(&(nns::hidfw::gfx::Graphics::GetInstance().GetGraphicsSystem().GetDevice()), info);
        nns::hidfw::gfx::Graphics::GetInstance().GetGraphicsSystem().RegisterTextureViewSlot(&m_FontTextureSlot, m_FontTextureView);
        // フォントエンジンの初期化を行います
        m_StandardFont.Initialize(m_ApplicationHeap.Allocate(1024 * 1024 * 32), 1024 * 1024 * 32);
        m_StandardFont.LoadFont(m_StandardFontName, nn::pl::GetSharedFontAddress(nn::pl::SharedFontType_Standard), 0, nn::fontll::FontNameLengthMax);
        m_StandardFont.SetFont(m_StandardFontName);
        m_NintendoExtensionFont.Initialize(m_ApplicationHeap.Allocate(1024 * 1024 * 32), 1024 * 1024 * 32);
        m_NintendoExtensionFont.LoadFont(m_NintendoExtensionFontName, nn::pl::GetSharedFontAddress(nn::pl::SharedFontType_NintendoExtension), 0, nn::fontll::FontNameLengthMax);
        m_NintendoExtensionFont.SetFont(m_NintendoExtensionFontName);
        // デフォルトのフォントサイズをセットします
        m_FontSize = 18;
        // フォントサイズを設定します。
        m_StandardFont.SetScale(m_FontSize << 16, 0, 0, m_FontSize << 16);
        m_NintendoExtensionFont.SetScale(m_FontSize << 16, 0, 0, m_FontSize << 16);

        SetFlags(nn::fontll::ScalableFontEngine::Flags_NoEffect);
        SetBorderWidth(0);

        // 縁取り設定(縁取り描画を行わない場合はコメントアウトします)
        m_OutLineWidth = 0;
        //m_FontEngine.SetOutlineWidth(m_OutLineWidth);
        //m_FontEngine.SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedUnFilled);
        // グリフの配置用にフォント情報を取得して必要な情報を記憶しておきます。
        m_StandardFont.GetFontMetrics(&m_Metrics);

        m_AscentRatio = static_cast<float>(m_Metrics.os2WinAscent) / m_Metrics.metricsResolution;

        m_PrintTextCount = 0;
        m_BufferOffset = 0;
    }

    void FontSystem::SetPos(float x, float y) NN_NOEXCEPT
    {
        m_PrintPosition.x = x;
        m_PrintPosition.y = y;
    }

    void FontSystem::SetPosX(float x) NN_NOEXCEPT
    {
        m_PrintPosition.x = x;
    }

    void FontSystem::SetPosY(float y) NN_NOEXCEPT
    {
        m_PrintPosition.y = y;
    }

    nn::util::Float2 FontSystem::GetPos() const NN_NOEXCEPT
    {
        return m_PrintPosition;
    }

    float FontSystem::GetPosX() const NN_NOEXCEPT
    {
        return m_PrintPosition.x;
    }

    float FontSystem::GetPosY() const NN_NOEXCEPT
    {
        return m_PrintPosition.y;
    }

    void FontSystem::SetSize(uint32_t fontSize) NN_NOEXCEPT
    {
        m_FontSize = fontSize < FontSizeMin ? FontSizeMin : fontSize > FontSizeMax ? FontSizeMax : fontSize;
        m_StandardFont.SetScale(m_FontSize << 16, 0, 0, m_FontSize << 16);
        m_NintendoExtensionFont.SetScale(m_FontSize << 16, 0, 0, m_FontSize << 16);
    }

    int FontSystem::SetFlags(const nn::fontll::ScalableFontEngine::Flags flag) NN_NOEXCEPT
    {
        return m_StandardFont.SetFlags(flag);
    }

    int FontSystem::SetBorderWidth(const uint16_t width) NN_NOEXCEPT
    {
        return m_StandardFont.SetOutlineWidth(width);
    }

    FontHandle FontSystem::CreateFont(const uint32_t charCode) NN_NOEXCEPT
    {
        FontHandle handle;

        if (!CheckCacheFont(&handle, charCode))
        {
            bool isExtension = (charCode & 0xFF00) >= 0xE000 && (charCode & 0xFF00) <= 0xF900;
            auto& fontEngine = isExtension ? m_NintendoExtensionFont : m_StandardFont;
            NN_ASSERT(fontEngine.CheckGlyphExist(charCode));

            // 登録されていないフォントの場合、新規に書き込みを行います
            FontData fontData;
            //NN_ASSERT_LESS_EQUAL(m_FontData.size(), nns::hidfw::gfx::FontSystem::CharCountMax);
            nn::fontll::GlyphMap* pMap = fontEngine.AcquireGlyphmap(charCode, nn::fontll::FormatGrayMap8);
            NN_ASSERT(pMap != NULL);

            uint16_t fontHeight = 16;
            const auto fontSize = (handle >> 48) & 0xFF;
            for (; fontHeight < 256 && fontHeight < fontSize; fontHeight *= 2) {}

            bool isFoundLine = false;
            uint16_t writePosX = 0;
            uint16_t writePosY = 0;

            for (auto& itr : m_LineData)
            {
                if (itr.lineSize == fontHeight &&
                    itr.nextWritePos + pMap->width < TextureWidth)
                {
                    isFoundLine = true;
                    writePosX = itr.nextWritePos;
                    itr.nextWritePos += pMap->width;
                    break;
                }
                writePosY += itr.lineSize;
            }
            if (!isFoundLine)
            {
                //NN_ASSERT_LESS(writePosY + fontHeight, TextureHeight);
                isFoundLine = true;
                LineData data;
                data.lineSize = fontHeight;
                data.nextWritePos = pMap->width;
                m_LineData.push_back(data);
            }

            fontData.glyp = *pMap;
            fontData.code = charCode;
            fontData.xPos = writePosX;
            fontData.yPos = writePosY;

            // 書き込みを開始します
            auto buffer = m_FontTextureBuffer.Map<uint8_t>();
            {
                memcpy(&buffer[m_BufferOffset], (void*)pMap->bits, pMap->width * pMap->height);
                m_FontTextureBuffer.FlushMappedRange(m_BufferOffset, pMap->width * pMap->height);
            }
            m_FontTextureBuffer.Unmap();
            //auto buffer = textureBuffer->Map<uint8_t>();
            //{
            //    for (int y = 0; y <  pMap->height; ++y)
            //    {
            //        auto offset = writePosY + (y * TextureWidth) + writePosX;
            //        textureBuffer->FlushMappedRange(offset, pMap->width);
            //        memcpy(&buffer[offset], (void*)pMap->bits[y * pMap->width], pMap->width);
            //    }
            //}
            //textureBuffer->Unmap();

            nn::gfx::BufferTextureCopyRegion region;
            region.SetDefault();
            region.EditTextureCopyRegion().SetOffsetU(writePosX);
            region.EditTextureCopyRegion().SetOffsetV(writePosY);
            region.EditTextureCopyRegion().SetWidth(pMap->width);
            region.EditTextureCopyRegion().SetHeight(pMap->height);
            region.SetBufferImageWidth(pMap->width);
            region.SetBufferImageHeight(pMap->height);
            region.SetBufferOffset(m_BufferOffset);

            nns::hidfw::gfx::Graphics::GetInstance().GetGraphicsSystem().GetCommandBuffer().CopyBufferToImage(
                &m_FontTexture, &m_FontTextureBuffer, region
            );
            m_BufferOffset += pMap->width * pMap->height;

            m_FontData.insert(std::pair<FontHandle, FontData>(handle, fontData));
            fontEngine.ReleasesGlyph(pMap);
        }
        return handle;
    }

    bool FontSystem::CheckCacheFont(FontHandle* pOutHandle, const uint32_t charCode) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutHandle);
        FontHandle handle =
            (0xFFFFFFFF & static_cast<uint64_t>(charCode)) |
            ((0x0000FFFF & (static_cast<uint64_t>(m_OutLineWidth))) << 32) |
            ((0x00FF & (static_cast<uint64_t>(m_FontSize))) << 48);
        auto result = m_FontData.find(handle) != m_FontData.end();
        *pOutHandle = handle;

        return result;
    }

    void FontSystem::PrintChar(const uint32_t ch) NN_NOEXCEPT
    {
        if (ch == U'\n')
        {
            m_PrintPosition.y += static_cast<float>(m_FontSize) + std::ceilf(m_FontSize * 0.2f);
            return;
        }

        if (m_PrintTextCount >= PrintTextCountMax)
        {
            return;
        }
        auto handle = CreateFont(ch);

        const auto& fontData = m_FontData.at(handle);
        const auto& glyp = fontData.glyp;
        const auto fontSize = static_cast<float>(0x00FF & (handle >> 48));

        m_pIndexBuffer[m_PrintTextCount * 6 + 0] = m_PrintTextCount * 4;
        m_pIndexBuffer[m_PrintTextCount * 6 + 1] = m_PrintTextCount * 4;
        m_pIndexBuffer[m_PrintTextCount * 6 + 2] = m_PrintTextCount * 4 + 1;
        m_pIndexBuffer[m_PrintTextCount * 6 + 3] = m_PrintTextCount * 4 + 2;
        m_pIndexBuffer[m_PrintTextCount * 6 + 4] = m_PrintTextCount * 4 + 3;
        m_pIndexBuffer[m_PrintTextCount * 6 + 5] = m_PrintTextCount * 4 + 3;

        nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(
            -1.f + (2.f * (m_PrintPosition.x + glyp.loX)) / 1280.f,
            1.f - (2.f * (m_PrintPosition.y + (fontSize - glyp.hiY))) / 720.f);

        nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(
            std::ceilf(2.f * static_cast<float>(glyp.width)) / 1280.f,
            std::ceilf(-2.f * static_cast<float>(glyp.height)) / 720.f
        );
        m_pVertexBuffer[m_PrintTextCount * 4 + 0] = nn::util::MakeFloat3(pos.x, pos.y, 0.f);
        m_pVertexBuffer[m_PrintTextCount * 4 + 1] = nn::util::MakeFloat3(pos.x, pos.y + size.y, 0.f);
        m_pVertexBuffer[m_PrintTextCount * 4 + 2] = nn::util::MakeFloat3(pos.x + size.x, pos.y, 0.f);
        m_pVertexBuffer[m_PrintTextCount * 4 + 3] = nn::util::MakeFloat3(pos.x + size.x, pos.y + size.y, 0.f);

        pos = nn::util::MakeFloat2(
            static_cast<float>(fontData.xPos) / static_cast<float>(TextureWidth),
            static_cast<float>(fontData.yPos) / static_cast<float>(TextureHeight)
        );
        size = nn::util::MakeFloat2(
            static_cast<float>(glyp.width) / static_cast<float>(TextureWidth),
            static_cast<float>(glyp.height) / static_cast<float>(TextureHeight)
        );
        m_pUvBuffer[m_PrintTextCount * 4 + 0] = nn::util::MakeFloat2(pos.x, pos.y);
        m_pUvBuffer[m_PrintTextCount * 4 + 1] = nn::util::MakeFloat2(pos.x, pos.y + size.y);
        m_pUvBuffer[m_PrintTextCount * 4 + 2] = nn::util::MakeFloat2(pos.x + size.x, pos.y);
        m_pUvBuffer[m_PrintTextCount * 4 + 3] = nn::util::MakeFloat2(pos.x + size.x, pos.y + size.y);

        const auto color = nn::util::MakeFloat4(
            static_cast<float>(gDrawer.GetColor().GetR()) / 255.f,
            static_cast<float>(gDrawer.GetColor().GetG()) / 255.f,
            static_cast<float>(gDrawer.GetColor().GetB()) / 255.f,
            static_cast<float>(gDrawer.GetColor().GetA()) / 255.f
        );

        for (int i = 0; i < 4; ++i)
        {
            m_pColorBuffer[m_PrintTextCount * 4 + i] = color;
        }

        m_PrintPosition.x += glyp.idX;
        ++m_PrintTextCount;
    }

    int FontSystem::Print(const char* str, ...) NN_NOEXCEPT
    {
        static char buf[PrintTextCountMax];
        static char32_t convStr[PrintTextCountMax];

        va_list vaList;
        va_start(vaList, str);
        auto length = vsnprintf(buf, PrintTextCountMax, str, vaList);
        va_end(vaList);

        if (
            (length > 0) &&
            (nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&length, buf) == nn::util::CharacterEncodingResult_Success) &&
            (nn::util::ConvertStringUtf8ToUtf32(convStr, PrintTextCountMax, buf) == nn::util::CharacterEncodingResult_Success)
            )
        {
            auto baseXPos = m_PrintPosition.x;

            for (int i = 0; i < length; ++i)
            {
                const auto ch = static_cast<uint32_t>(convStr[i]);
                if (ch == U'\n')
                {
                    m_PrintPosition.x = baseXPos;
                }
                PrintChar(ch);
            }
            return length;
        }
        else
        {
            return 0;
        }
    }

    void FontSystem::Flush() NN_NOEXCEPT
    {
        m_PrintTextCount = 0;
    }

    const uint32_t* FontSystem::GetIndexBuffer() const NN_NOEXCEPT { return m_pIndexBuffer; }
    const nn::util::Float3* FontSystem::GetVertexBuffer() const NN_NOEXCEPT { return m_pVertexBuffer; }
    const nn::util::Float2* FontSystem::GetUvBuffer() const NN_NOEXCEPT { return m_pUvBuffer; }
    const nn::util::Float4* FontSystem::GetColorBuffer() const NN_NOEXCEPT { return m_pColorBuffer; }

    size_t FontSystem::GetIndexBufferSize() const NN_NOEXCEPT { return sizeof(uint32_t) * GetIndexBufferCount(); }
    size_t FontSystem::GetVertexBufferSize() const NN_NOEXCEPT { return sizeof(nn::util::Float3) * GetVertexBufferCount(); }
    size_t FontSystem::GetUvBufferSize() const NN_NOEXCEPT { return sizeof(nn::util::Float2) * GetUvBufferCount(); }
    size_t FontSystem::GetColorBufferSize() const NN_NOEXCEPT { return sizeof(nn::util::Float4) * GetColorBufferCount(); }

    int FontSystem::GetIndexBufferCount() const NN_NOEXCEPT { return m_PrintTextCount * 6; }
    int FontSystem::GetVertexBufferCount() const NN_NOEXCEPT { return m_PrintTextCount * 4; }
    int FontSystem::GetUvBufferCount() const NN_NOEXCEPT { return m_PrintTextCount * 4; }
    int FontSystem::GetColorBufferCount() const NN_NOEXCEPT { return m_PrintTextCount * 4; }

    nn::gfx::DescriptorSlot* FontSystem::GetDescriptorSlot() NN_NOEXCEPT { return &m_FontTextureSlot; }

    //----------------------------------------------------------------
    //! @brief 描画領域を指定してテキストの描画を行います
    //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
    //----------------------------------------------------------------
    void FontSystem::DrawText() NN_NOEXCEPT
    {
        if (m_PrintTextCount > 0)
        {
            nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(gGraphics.GetGraphicsSystem().GetPrimitiveRenderer());
            nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

            if (mesh.Initialize(
                pRenderer->GetGpuBuffer(),
                GetVertexBufferCount(),
                GetIndexBufferCount(),
                (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Uv | nns::gfx::PrimitiveRenderer::VertexFormat_Color)) == false)
            {
                return;
            }

            uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
            memcpy(pIndexData, GetIndexBuffer(), GetIndexBufferSize());

            nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
            memcpy(pPos, GetVertexBuffer(), GetVertexBufferSize());

            nn::util::Float2* pUv = static_cast<nn::util::Float2*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
            memcpy(pUv, GetUvBuffer(), GetUvBufferSize());

            nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
            memcpy(pColor, GetColorBuffer(), GetColorBufferSize());

            pRenderer->SetColor(nn::util::Color4u8::White());

            pRenderer->DrawUserMesh(
                &(gGraphics.GetGraphicsSystem().GetCommandBuffer()),
                nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleStrip,
                &mesh,
                m_FontTextureSlot, gDrawer.GetSamplerDescriptorSlot());
        }
        Flush();
    }

    void FontSystem::DrawFontTexture() NN_NOEXCEPT
    {
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(gGraphics.GetGraphicsSystem().GetPrimitiveRenderer());
        nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

        if (mesh.Initialize(pRenderer->GetGpuBuffer(), 4, 4,
            (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Uv)) == false)
        {
            return;
        }

        uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();
        pIndexData[0] = 0;
        pIndexData[1] = 1;
        pIndexData[2] = 2;
        pIndexData[3] = 3;

        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
        pPos[0] = nn::util::MakeFloat3(-1, -1, 0);
        pPos[1] = nn::util::MakeFloat3(1, -1, 0);
        pPos[2] = nn::util::MakeFloat3(-1, 1, 0);
        pPos[3] = nn::util::MakeFloat3(1, 1, 0);

        nn::util::Float2* pUv = static_cast<nn::util::Float2*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
        pUv[0] = nn::util::MakeFloat2(0.0f, 0.2f);
        pUv[1] = nn::util::MakeFloat2(0.2f, 0.2f);
        pUv[2] = nn::util::MakeFloat2(0.0f, 0.0f);
        pUv[3] = nn::util::MakeFloat2(0.2f, 0.0f);

        pRenderer->DrawUserMesh(
            &(gGraphics.GetGraphicsSystem().GetCommandBuffer()),
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleStrip,
            &mesh,
            m_FontTextureSlot, gDrawer.GetSamplerDescriptorSlot());
        Flush();
    }

}}}
