﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>
#include <nn/util/util_FormatString.h>

#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>

namespace {

glv::GLV s_GlvRootView;

void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

void SetupPeripherals() NN_NOEXCEPT
{
    const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

glv::Window* g_pWindow = nullptr;
const int WindowStackSize = 0x40000;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_WindowStack[WindowStackSize];
nn::os::ThreadType g_WindowThread;

void WindowFunction(void* arg) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    g_pWindow->setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

// リソースを初期化して Window スレッドを立ち上げる
void InitializeGlv() NN_NOEXCEPT
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    g_pWindow = new glv::Window(width, height, "Main Window");

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_WindowThread, WindowFunction, nullptr, g_WindowStack, WindowStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread())));
    nn::os::StartThread(&g_WindowThread);
}

void FinalizeGlv() NN_NOEXCEPT
{
    // TODO: 正しい終了処理を確認する。
    //glv::Application::quit();
    //nn::os::WaitThread(&g_WindowThread);
    //nn::os::DestroyThread(&g_WindowThread);
    if ( g_pWindow != nullptr )
    {
        delete g_pWindow;
    }
}

void AddItemToView(glv::Label& item) NN_NOEXCEPT
{
    s_GlvRootView << item;
}

void AddItemToView(glv::Button& item) NN_NOEXCEPT
{
    s_GlvRootView << item;
}
