﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/rid.h>
#include "../DevQuestMenu_Log.h"
#include "DevQuestMenu_SystemUpdateModeScene.h"

namespace
{
    //! ダウンロード状況のラベル
    static const std::map<nn::rid::SystemUpdateProgress::State, std::string> UpdateStateLabel =
    {
        { nn::rid::SystemUpdateProgress::State::DoNothing, "DoNothing" },
        { nn::rid::SystemUpdateProgress::State::Checking, "Checking" },
        { nn::rid::SystemUpdateProgress::State::Downloading, "Downloading" },
        { nn::rid::SystemUpdateProgress::State::Applying, "Applying" },
        { nn::rid::SystemUpdateProgress::State::Completed, "Completed" },
        { nn::rid::SystemUpdateProgress::State::Failed, "Failed" },
        { nn::rid::SystemUpdateProgress::State::NeedNoUpdate, "NeedNoUpdate" },
    };
}

namespace nn { namespace devquestmenu {

    /**
     * @brief       本体更新モード起動時処理を行います
     *
     * @detail      本体更新を実行するスレッドを生成・実行します。
     */
    SystemUpdateModeScene::SystemUpdateModeScene(RootSurfaceContext* pRootSurface) NN_NOEXCEPT
        : ModeSceneCommon("System Update Mode", pRootSurface)
    {
        InitializeView();

        //! 本体更新スレッドの生成
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
        {
            auto self = reinterpret_cast<SystemUpdateModeScene*>(p);
            self->m_SystemUpdateResult = self->m_SystemUpdater.Execute();
            self->m_IsSystemUpdateDone = true;
        }
        , this, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority));

        //! 本体更新開始
        nn::os::StartThread(&m_UpdateExecutionThread);
    };

    /**
     * @brief       本体更新モード終了時処理を行います。
     *
     */
    SystemUpdateModeScene::~SystemUpdateModeScene() NN_NOEXCEPT
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }

    /**
     * @brief       本体更新をポーリングします
     *
     */
    void SystemUpdateModeScene::OnLoop() NN_NOEXCEPT
    {
        rid::SystemUpdateProgress progress = m_SystemUpdater.GetProgress();
        m_SystemUpdateStatus = MakeLabel(UpdateStateLabel.at(progress.state).c_str(), m_StatusLabelSpec);

        if (progress.state == rid::SystemUpdateProgress::State::Downloading)
        {
            QUESTMENU_LOG("Downloading: %lld / %lld\n", progress.loaded, progress.total);
            m_SystemUpdateProgress.get()->setValue(static_cast<double>(progress.loaded) / static_cast<double>(progress.total));
        }
        else
        {
            QUESTMENU_LOG("%s\n", UpdateStateLabel.at(progress.state).c_str());
        }
    };

    /**
     * @brief       アプリケーション更新モードに遷移できるかを判定します
     *
     * @param[in]   displayModeTime     更新開始時間の設定
     *
     */
    ModeType SystemUpdateModeScene::DetermineNextMode(const DisplayModeTime& displayModeTime) NN_NOEXCEPT
    {
        NN_UNUSED(displayModeTime);

        if (!m_IsSystemUpdateDone)    //! 本体更新が完了していない
        {
            return ModeType_NonChange;
        }
        else
        {
            if (m_SystemUpdateResult.IsFailure()) //! システムアップデートに失敗した
            {
                QUESTMENU_LOG("Failed as result 0x%08x\n", m_SystemUpdateResult.GetInnerValueForDebug());
                return ModeType_SleepMode;
            }
            else
            {
                //! 本体更新完了時は要メニュー更新フラグを立てて本体再起動する
                rid::SystemUpdateProgress progress = m_SystemUpdater.GetProgress();
                if (progress.state == rid::SystemUpdateProgress::State::Completed)
                {
                    m_pRootSurface->SetMenuUpdateRequired(true); //! 要メニュー更新フラグを立てる
                    m_pRootSurface->SaveSettingsAndReboot(); //! 本体再起動
                }

                return ModeType_MenuUpdateMode;
            }
        }
    };

    ModeType SystemUpdateModeScene::GetSuccessNextMode() NN_NOEXCEPT
    {
        if (!m_IsSystemUpdateDone)    //! 本体更新が完了していない
        {
            m_SystemUpdateStatus = MakeLabel("Now System Updating. Please Wait!", m_StatusLabelSpec);
            return ModeType_NonChange;
        }
        else
        {
            return ModeType_MenuUpdateMode;
        }
    };

    void SystemUpdateModeScene::InitializeView() NN_NOEXCEPT
    {
        //! 更新状態ラベル
        m_SystemUpdateStatus = MakeLabel("System Update Start", m_StatusLabelSpec);

        //! 進捗バー
        std::unique_ptr<glv::Slider> slider(new glv::Slider(m_ProgressSliderRect));
        slider.get()->disable(glv::Property::FocusToTop | glv::Property::HitTest);
        *this << slider.get();
        m_SystemUpdateProgress = std::move(slider);
    };
}}
