﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/applet/applet_IdlePolicyApi.h>
#include <nn/bpc/bpc_WakeupConfig.h>
#include <nn/oe/oe_NotificationMessage.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>
#include "../DevQuestMenu_Log.h"
#include "../DevQuestMenu_DisplayModeTime.h"
#include "DevQuestMenu_SleepModeScene.h"

namespace nn { namespace devquestmenu {

    SleepModeScene::SleepModeScene(RootSurfaceContext* pRootSurface) NN_NOEXCEPT
        : ModeSceneCommon("Sleep Mode", pRootSurface)
    {
        m_pRootSurface->SaveSettings(); //! メニュー設定を保存する
        SleepWithWakeUpTimer();
        m_IsSleepFinished = true;
    };

    void SleepModeScene::OnLoop() NN_NOEXCEPT
    {
    };

    ModeType SleepModeScene::DetermineNextMode(const DisplayModeTime& displayModeTime) NN_NOEXCEPT
    {
        if (m_IsSleepFinished)
        {
            //! スリープ状態が終われば本体再起動する
            m_pRootSurface->SaveSettingsAndReboot();
        }

        return ModeType_NonChange;
    };

    ModeType SleepModeScene::GetSuccessNextMode() NN_NOEXCEPT
    {
        //! 再起動
        m_pRootSurface->SaveSettingsAndReboot();
        return ModeType_NonChange;
    };

    void SleepModeScene::SleepWithWakeUpTimer() NN_NOEXCEPT
    {
        nn::time::CalendarTime currentTime = NnTimeWrapper::GetCurrentTime();
        int64_t targetTime = NnTimeWrapper::GetSecondForTargetTime(m_pRootSurface->GetDisplayModeTime().startTime, currentTime);

        NN_ASSERT(targetTime >= 0);

        //! タイマーの最小時間に丸め込み
        if (targetTime < nn::bpc::MinWakeupIntervalInSeconds)
        {
            targetTime = nn::bpc::MinWakeupIntervalInSeconds;
        }

        nn::TimeSpan time = nn::TimeSpan::FromSeconds(targetTime);

        //! wake up タイマをセットする
        int timerHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bpc::CreateWakeupTimer(
            &timerHandle, time, nn::bpc::WakeupTimerType::WakeupTimerType_FullWakeup
        ));

        //! 通知設定
        nn::oe::SetResumeNotificationEnabled(true);

        //! 5 秒後にスリープ
        const int autoSleepTime = 5;
        nn::applet::OverrideAutoSleepTimeAndDimmingTime(autoSleepTime, autoSleepTime, autoSleepTime, autoSleepTime);

        QUESTMENU_LOG("Quest will go into sleep in %d seconds and wake up in %lld seconds\n", autoSleepTime, targetTime + 5);

        //! 起床を待機する
        while (nn::oe::MessageResume != nn::oe::PopNotificationMessage())
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        QUESTMENU_LOG("Quest has waked up\n");

        //! スリープ前の設定に戻す
        nn::applet::OverrideAutoSleepTimeAndDimmingTime(0, 0, 0, 0);
    };
}}
