﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>
#include "DevQuestMenu_Log.h"
#include "DevQuestMenu_RidFileIo.h"

namespace nn { namespace devquestmenu {

    /**
     * @brief       file を読み込みます
     *
     * @param[out]  pBuffer     バッファ
     * @param[in]   bufferSize  バッファサイズ
     * @param[in]   filePath    ファイルパス
     */
    bool RidFileIo::ReadFile(nn::Bit8* pBuffer, int bufferSize, const char* filePath) NN_NOEXCEPT
    {
        //! ファイルを Open する
        nn::fs::FileHandle fileHandle;
        nn::Result result = nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode::OpenMode_Read);

        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            //! ファイルが存在しない場合 false を返す
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        int64_t readFileSize;

        //! ファイルサイズの取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&readFileSize, fileHandle));
        QUESTMENU_LOG("%s file size: %lld\n", filePath, readFileSize);
        NN_ABORT_UNLESS(readFileSize <= bufferSize);

        //! バッファクリア
        memset(pBuffer, 0, bufferSize);

        //! ファイルの読み取り
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, pBuffer, readFileSize));
        pBuffer[bufferSize - 1] = '\0';
        return true;
    };
}}

