﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstring>
#include "common.h"
#include "DevQuestMenu_RapidJsonConfig.h"

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(pop)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#include <rapidjson/document.h>
#include <rapidjson/prettywriter.h>
#include <rapidjson/writer.h>

//! expression が false になった時、return false する
#define FAILURE_UNLESS(expression) \
    do { auto _r = (expression); if (!_r) return false; } while (NN_STATIC_CONDITION(false))

namespace nn { namespace devquestmenu {

    class RapidJsonParser
    {
    public:

        //! "key" : { pStringObject[0], pStringObject[1], pStringObject[2], ... }
        static const std::string CreateJsonObject(const std::string& key, const std::vector<std::string>& pStringObject, int num) NN_NOEXCEPT
        {
            std::string buffer = "\"" + key + "\":{";
            for (int i = 0; i < num; i++)
            {
                buffer += (i != 0 ? ",": "") + pStringObject[i];
            }
            buffer += "}";
            return buffer;
        }

        //! "key" : [ pStringArray[0], pStringArray[1], pStringArray[2], ... ]
        static const std::string CreateJsonArray(const std::string& key, const std::vector<std::string>& pStringArray, int num) NN_NOEXCEPT
        {
            nne::rapidjson::StringBuffer buffer;
            nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(buffer);
            writer.StartArray();
            for (int i = 0; i < num; i++)
            {
                writer.String(pStringArray[i].c_str());
            }
            writer.EndArray();
            std::string value = "\"" + key + "\":" + buffer.GetString();
            return value;
        }

        //! JSON Text
        static bool CreateJsonText(std::string* pText, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            nne::rapidjson::StringBuffer buffer;
            nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(buffer);
            FAILURE_UNLESS(document.Accept(writer));
            *pText = buffer.GetString();
            return true;
        }

        //! Pretty JSON Text
        static bool CreatePrettyJsonText(std::string* pText, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            nne::rapidjson::StringBuffer buffer;
            nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
            FAILURE_UNLESS(document.Accept(writer));
            *pText = buffer.GetString();
            return true;
        }

        //! Bool
        static bool GetBoolValue(bool* pOutValue, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            *pOutValue = document[cKey].GetBool();
            return exists;
        }

        static bool GetBoolValue(bool* pOutValue, const std::string& key, const nne::rapidjson::Value& jsonValue) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            for (nne::rapidjson::Value::ConstMemberIterator iterator = jsonValue.MemberBegin();
                iterator != jsonValue.MemberEnd(); iterator++)
            {
                if (iterator->name == cKey)
                {
                    *pOutValue = iterator->value.GetBool();
                    return true;
                }
            }

            return false;
        }

        //! Int
        static bool GetIntValue(int* pOutValue, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            *pOutValue = document[cKey].GetInt();
            return exists;
        }

        //! Uint
        static bool GetUintValue(uint32_t* pOutValue, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            *pOutValue = document[cKey].GetUint();
            return exists;
        }

        static bool GetIntValue(int* pOutValue, const std::string& key, const nne::rapidjson::Value& jsonValue) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            for (nne::rapidjson::Value::ConstMemberIterator iterator = jsonValue.MemberBegin();
                iterator != jsonValue.MemberEnd(); iterator++)
            {
                if (iterator->name == cKey)
                {
                    *pOutValue = iterator->value.GetInt();
                    return true;
                }
            }

            return false;
        }

        static bool GetUintValue(uint32_t* pOutValue, const std::string& key, const nne::rapidjson::Value& jsonValue) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            for (nne::rapidjson::Value::ConstMemberIterator iterator = jsonValue.MemberBegin();
                iterator != jsonValue.MemberEnd(); iterator++)
            {
                if (iterator->name == cKey)
                {
                    *pOutValue = iterator->value.GetUint();
                    return true;
                }
            }

            return false;
        }

        //! String
        static bool GetStringValue(std::string* pOutValue, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            *pOutValue = document[cKey].GetString();
            return exists;
        }

        static bool GetStringValue(std::string* pOutValue, const std::string& key, const nne::rapidjson::Value& jsonValue) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            for (nne::rapidjson::Value::ConstMemberIterator iterator = jsonValue.MemberBegin();
                iterator != jsonValue.MemberEnd(); iterator++)
            {
                if (iterator->name == cKey)
                {
                    *pOutValue = iterator->value.GetString();
                    return true;
                }
            }
            return false;
        }

        //! String Array
        static bool GetStringArrayLength(int* pOutArrayLength, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            const nne::rapidjson::Value& valueOnJson = document[cKey];
            *pOutArrayLength = valueOnJson.Size();
            return exists;
        }

        static bool GetStringArray(std::string* pOutArrayValue, int length, const std::string& key, const nne::rapidjson::Document& document) NN_NOEXCEPT
        {
            const char* cKey = key.c_str();
            bool exists = document.HasMember(cKey);
            FAILURE_UNLESS(exists);
            const nne::rapidjson::Value& valueOnJson = document[cKey];
            for (int i = 0; i < length; i++)
            {
                pOutArrayValue[i] = valueOnJson[i].GetString();
            }
            return exists;
        }
    };
}}
