﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iomanip>
#include <sstream>
#include <nn/time/time_ApiForMenu.h>

namespace nn { namespace devquestmenu {

    class NnTimeWrapper
    {
    public:

        //! 時刻 0 埋め用
        static const std::string ConvertToStringHour(const nn::time::CalendarTime& time) NN_NOEXCEPT
        {
            std::ostringstream stream;
            stream << std::setfill('0') << std::setw(2) << std::to_string(time.hour);
            return stream.str();
        };

        static const std::string ConvertToStringMinute(const nn::time::CalendarTime& time) NN_NOEXCEPT
        {
            std::ostringstream stream;
            stream << std::setfill('0') << std::setw(2) << std::to_string(time.minute);
            return stream.str();
        };

        //! hour, minute -> Base Calender Time
        static const nn::time::CalendarTime ConvertBaseCalenderTime(int8_t hour, int8_t minute) NN_NOEXCEPT
        {
            return { BaseYear, BaseMonth, BaseDay, hour, minute, 0 };
        }

        //! Calender Time -> Base Calender Time
        static const nn::time::CalendarTime ConvertBaseCalenderTime(const nn::time::CalendarTime& time) NN_NOEXCEPT
        {
            return { BaseYear, BaseMonth, BaseDay, time.hour, time.minute, time.second };
        }

        //! Get Base Current Time
        static const nn::time::CalendarTime GetCurrentTime() NN_NOEXCEPT
        {
            nn::time::PosixTime currentTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&currentTime)); // ローカルタイムを取得する
            return ConvertBaseCalenderTime(currentTime);
        };

        //! Calender Time Adjust
        static void AddHour(nn::time::CalendarTime* pTime, int addValue) NN_NOEXCEPT
        {
            pTime->hour += addValue;
            pTime->hour = Rotate<int8_t>(pTime->hour, 24);
        }

        static void AddMinute(nn::time::CalendarTime* pTime, int addValue) NN_NOEXCEPT
        {
            pTime->minute += addValue;
            pTime->minute = Rotate<int8_t>(pTime->minute, 60);
        }

        //! Get Timer Seconds
        static int64_t GetSecondForTargetTime(const nn::time::CalendarTime& targetTime, const nn::time::CalendarTime& currentTime) NN_NOEXCEPT
        {
            //! current time > target time の場合、次の日の target time までの時間を求めます
            int64_t period = ((targetTime.hour * 60LL + targetTime.minute) - (currentTime.hour * 60LL + currentTime.minute)) * 60LL - currentTime.second;
            return period >= 0 ? period : 24LL * 60LL * 60LL + period;
        }

    private:

        //! Posix Time -> Base Calender Time
        static const nn::time::CalendarTime ConvertBaseCalenderTime(const nn::time::PosixTime& posixTime) NN_NOEXCEPT
        {
            nn::time::CalendarTime calenderTime = nn::time::ToCalendarTimeInUtc(posixTime);
            return ConvertBaseCalenderTime(calenderTime);
        }

        template<typename T>
        static T Rotate(T input, T max) NN_NOEXCEPT
        {
            //! min = 0

            NN_STATIC_ASSERT(std::is_integral<T>::value);
            return input >= 0 ? input % max : max - (std::abs(input) % max);
        }

        static const int16_t BaseYear = 1970;
        static const int8_t BaseMonth = 1;
        static const int8_t BaseDay = 1;
    };

    class DisplayModeTime
    {
    public:
        nn::time::CalendarTime startTime;
        nn::time::CalendarTime endTime;

        DisplayModeTime(int8_t startHour, int8_t startMinute, int8_t endHour, int8_t endMinute) NN_NOEXCEPT
        {
            startTime = NnTimeWrapper::ConvertBaseCalenderTime(startHour, startMinute);
            endTime = NnTimeWrapper::ConvertBaseCalenderTime(endHour, endMinute);
        };

        bool IsDuringDisplayModeTime() const NN_NOEXCEPT
        {
            nn::time::CalendarTime currentTime = NnTimeWrapper::GetCurrentTime();
            int startTimeMinute = startTime.hour * 60 + startTime.minute;
            int endTimeMinute = endTime.hour * 60 + endTime.minute;
            int currentTimeMinute = currentTime.hour * 60 + currentTime.minute;

            if (startTimeMinute <= endTimeMinute)
            {
                return startTimeMinute <= currentTimeMinute && currentTimeMinute < endTimeMinute;
            }
            else
            {
                return startTimeMinute <= currentTimeMinute || currentTimeMinute < endTimeMinute;
            }
        };
    };
}}
