﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <sstream>
#include <iostream>
#include <nn/applet/applet_ApplicationControlForQuest.h>
#include <nn/ncm/ncm_ApplicationInfo.h>
#include <nn/oe.h>
#include <nn/rid.h>
#include "DevQuestMenu_AssetMeta.h"
#include "DevQuestMenu_DebugSettings.h"
#include "DevQuestMenu_Log.h"
#include "DevQuestMenu_RapidJsonParser.h"

namespace nn { namespace devquestmenu {

    namespace
    {
        applet::ApplicationAttributeForQuest GetApplicationAttributeSetting(uint32_t contentGamePlayTimer) NN_NOEXCEPT
        {
            applet::ApplicationAttributeForQuest timerAttribute = { 0, 0 };

            // SD カードの設定ファイルを読み込み体験終了タイマーの設定を変更する(簡易 QuestSettings 機能)
            bool isPlayableTimerEnabled;
            // 設定ファイルを読み込めた場合、設定を反映する
            if (ReadPlayableTimerSettingsForDebug(&isPlayableTimerEnabled))
            {
                timerAttribute.playableTime = isPlayableTimerEnabled ? contentGamePlayTimer : 0;
            }
            // 読み込めなかった場合、デフォルトの設定を使う
            else
            {
                timerAttribute.playableTime = contentGamePlayTimer;
            }

            // SD カードの設定ファイルを読み込み無操作時間タイマーの設定を変更する(簡易 QuestSettings 機能)
            bool isIdleDetectionTimerEnabled;
            uint32_t idleDetectionTime;
            // 設定ファイルを読み込めた場合、設定を反映する
            if (ReadIdleDetectionTimerSettingsForDebug(&isIdleDetectionTimerEnabled, &idleDetectionTime))
            {
                timerAttribute.idleDetectionTime = isIdleDetectionTimerEnabled ? idleDetectionTime : 0;
            }
            // 読み込めなかった場合、デフォルトの設定を使う
            else
            {
                timerAttribute.idleDetectionTime = 120;
            }

            return timerAttribute;
        }
    }

    AssetMeta::AssetMeta(const char* buffer) NN_NOEXCEPT
    {
        InitializeAssetMeta(buffer);
    }

    void AssetMeta::InitializeAssetMeta(const char* buffer) NN_NOEXCEPT
    {
        nne::rapidjson::Document document;
        document.Parse(buffer);
        if (document.HasParseError())
        {
            QUESTMENU_LOG("parse error. error code: %d\n", document.GetParseError());
            return;
        }

        auto GetCountryBasisStringValue = [](CountryBasisString* outValue, const std::string& key, const nne::rapidjson::Document& document)
        {
            if (document.HasMember(key.c_str()))
            {
                const nne::rapidjson::Value& value = document[key.c_str()];
                int size = value.Size();

                for (int i = 0; i < size; i++)
                {
                    RapidJsonParser::GetStringValue(&outValue->jpJapanese, "JP_Japanese", value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usEnglish, "US_English", value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usFrench, "US_French", value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usSpanish, "US_Spanish", value[i]);
                    RapidJsonParser::GetStringValue(&outValue->euEnglish, "EU_English", value[i]);
                }
            }
        };

        auto GetCountryBasisStringValueByValue = [](CountryBasisString* outValue, const std::string& key, const nne::rapidjson::Value& value)
        {
            auto iter = value.FindMember(key.c_str());
            if (iter != value.MemberEnd())
            {
                int size = iter->value.Size();

                for (int i = 0; i < size; i++)
                {
                    RapidJsonParser::GetStringValue(&outValue->jpJapanese, "JP_Japanese", iter->value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usEnglish, "US_English", iter->value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usFrench, "US_French", iter->value[i]);
                    RapidJsonParser::GetStringValue(&outValue->usSpanish, "US_Spanish", iter->value[i]);
                    RapidJsonParser::GetStringValue(&outValue->euEnglish, "EU_English", iter->value[i]);
                }
            }
        };

        auto GetStringValueList = [](std::vector<std::string>* outValue, const std::string& key, const nne::rapidjson::Document& document)
        {
            if (document.HasMember(key.c_str()))
            {
                const nne::rapidjson::Value& value = document[key.c_str()];
                int size = value.Size();

                for (int i = 0; i < size; i++)
                {
                    if (value[i].IsString())
                    {
                        outValue->push_back(value[i].GetString());
                    }
                }
            }
        };

        // TORIAEZU: 旧フォーマートの Asset 対応
        // きちんとした AssetDatabase は作れない
        if (document.HasMember("timer"))
        {
            QUESTMENU_LOG("Read old format asset.\n");

            RapidJsonParser::GetStringValue(&m_DemoApplicationId, "application_id", document);

            std::string timer;
            RapidJsonParser::GetStringValue(&timer, "timer", document);
            m_GamePlayTimer = strtol(timer.c_str(), nullptr, 10);

            GetCountryBasisStringValue(&m_Title, "title", document);
        }
        else
        {
            RapidJsonParser::GetIntValue(&m_Index, "index", document);
            RapidJsonParser::GetIntValue(&m_Version, "version", document);
            RapidJsonParser::GetStringValue(&m_AocApplicationId, "application_id", document);
            RapidJsonParser::GetStringValue(&m_DemoApplicationId, "program_id", document);
            GetCountryBasisStringValue(&m_Title, "title", document);
            GetStringValueList(&m_CategoryList, "categories", document);
            GetCountryBasisStringValue(&m_Box, "box", document);
            GetCountryBasisStringValue(&m_Movie, "movie", document);
            GetCountryBasisStringValue(&m_Description, "description", document);
            GetCountryBasisStringValue(&m_Publisher, "publisher", document);
            GetCountryBasisStringValue(&m_Developer, "developer", document);
            GetCountryBasisStringValue(&m_ReleaseDate, "release_date", document);
            GetCountryBasisStringValue(&m_Price, "price", document);

            if (document.HasMember("multiplayer"))
            {
                const nne::rapidjson::Value& multiplayer = document["multiplayer"];
                RapidJsonParser::GetBoolValue(&m_IsTvMultiPlayerSupported, "tv", multiplayer);
                RapidJsonParser::GetBoolValue(&m_IsTabletopMultiPlayerSupported, "tabletop", multiplayer);
                RapidJsonParser::GetBoolValue(&m_IsHandheldMultiPlayerSupported, "handheld", multiplayer);
            }

            GetStringValueList(&m_SuppurtedControllerList, "supported_controllers", document);

            RapidJsonParser::GetBoolValue(&m_IsMovieOnly, "movie_only", document);
            RapidJsonParser::GetUintValue(&m_GamePlayTimer, "game_play_timer", document);

            if (document.HasMember("ratings"))
            {
                const nne::rapidjson::Value& ratings = document["ratings"];
                int size = ratings.Size();
                for (int i = 0; i < size; i++)
                {
                    Rating rating;

                    RapidJsonParser::GetStringValue(&rating.ratingBoard, "rating_board", ratings[i]);
                    RapidJsonParser::GetIntValue(&rating.ratingValue, "rating_value", ratings[i]);
                    RapidJsonParser::GetBoolValue(&rating.isRatingPending, "rating_pending", ratings[i]);

                    int j = 0;
                    for (nne::rapidjson::Value::ConstMemberIterator iter = ratings[i].MemberBegin();
                        iter != ratings[i].MemberEnd(); iter++, j++)
                    {
                        if (iter->name == "rating_descriptor_list")
                        {
                            for (int k = 0; k < iter->value.Size(); k++)
                            {
                                CountryBasisString ratingDescripter;
                                GetCountryBasisStringValueByValue(&ratingDescripter, "rating_descriptor", iter->value[k]);
                                rating.ratingDescripterList.push_back(ratingDescripter);
                            }
                        }
                    }

                    m_RatingList.push_back(rating);
                }
            }
        }
        RapidJsonParser::CreatePrettyJsonText(&m_JsonFormat, document);
    } // NOLINT(impl/function_size)

    /**
     * @brief       AssetMeta に設定された Application ID の Application を起動します。
     */
    void AssetMeta::LaunchApplication(bool isLaunchDelayed) NN_NOEXCEPT
    {
        LogOutAssetMetaValue();
        nn::ncm::ApplicationId id = { std::stoull(m_DemoApplicationId, nullptr, 16) };
        applet::ApplicationAttributeForQuest timerAttribute = GetApplicationAttributeSetting(m_GamePlayTimer);

        if (isLaunchDelayed)
        {
            nn::oe::EnterExitRequestHandlingSection();
        }

        QUESTMENU_LOG("RequestToLaunchApplication: playableTime=%u(s), idleDetectionTime=%u(s)\n", timerAttribute.playableTime, timerAttribute.idleDetectionTime);
        nn::rid::RequestToLaunchApplication(id, &timerAttribute);

        if (isLaunchDelayed)
        {
            DelayLaunch();
        }
    }

    /**
     * @brief       体験版アプリの起動を 40 秒遅延させます（デバッグ用）
     */
    void AssetMeta::DelayLaunch() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto message = nn::oe::PopNotificationMessage();
            if (message == nn::oe::MessageExitRequest)
            {
                const int delayTime = 40;
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(delayTime));
                nn::oe::LeaveExitRequestHandlingSection();
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    /**
     * @brief       AssetMeta の値をログに表示します。
     */
    void AssetMeta::LogOutAssetMetaValue() NN_NOEXCEPT
    {
        QUESTMENU_LOG("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
        QUESTMENU_LOG("json:\n%s\n", m_JsonFormat.c_str());
        QUESTMENU_LOG("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
    }
}}
