﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../ui_IMenuItem.h"
#include "ui_MenuItemGeometry.h"

namespace ui{ namespace util{

    class FindFocusAcceptingItem
    {
    private:
        template<typename F>
        struct Context
        {
            float candidateDistance;
            std::shared_ptr<IMenuItem> pCandidate;

            std::shared_ptr<const IMenuItem> pCurrentFocused;
            F distanceFunction;
        };

    public:
        static float Distance(const std::shared_ptr<const IMenuItem>& a, const std::shared_ptr<const IMenuItem>& b) NN_NOEXCEPT
        {
            // TORIAEZU: 中心間のマンハッタン距離
            auto arect = MenuItemGeometry::GetGlobalRectangle(a);
            auto brect = MenuItemGeometry::GetGlobalRectangle(b);
            Position ca = {
                arect.position.x + arect.size.width  / 2,
                arect.position.y + arect.size.height / 2,
            };
            Position cb = {
                brect.position.x + brect.size.width  / 2,
                brect.position.y + brect.size.height / 2,
            };

            return std::abs(ca.x - cb.x) + std::abs(ca.y - cb.y);
        }

        // pItem またはその子のうちフォーカスを受け取るものを見つける
        // フォーカスを受け取るものがない場合、 nullptr を返す。
        // 複数のアイテムがフォーカスを受け取れる場合、 distanceFunction(p, pCurrentFocused) の値が最も小さいものが選ばれる。
        // ただし、 pCurrentFocused == nullptr の場合、最初に見つけたアイテムを返す。
        //
        // 表示されないアイテムはフォーカスを受け取る対象から除外される。
        //
        // float distanceFunction(const std::shared_ptr<const IMenuItem>&, const std::shared_ptr<const IMenuItem>&)
        template<typename F>
        static std::shared_ptr<IMenuItem> Find(
            const std::shared_ptr<IMenuItem>& pItem,
            const std::shared_ptr<const IMenuItem>& pCurrentFocused,
            F distanceFunction
        ) NN_NOEXCEPT
        {
            if(!pItem)
            {
                return nullptr;
            }

            auto crop = Aabb::FromRectangle(MenuItemGeometry::GetGlobalRectangle(pItem));
            return Find(pItem, pCurrentFocused, crop, distanceFunction);
        }

        template<typename F>
        static std::shared_ptr<IMenuItem> Find(
            const std::shared_ptr<IMenuItem>& pItem,
            const std::shared_ptr<const IMenuItem>& pCurrentFocused,
            const Aabb& crop,
            F distanceFunction
        ) NN_NOEXCEPT
        {
            Context<F> context = {std::numeric_limits<float>::max(), nullptr, pCurrentFocused, distanceFunction};
            FindImpl(pItem, context, crop);
            return context.pCandidate;
        }

    private:

        // 確定した場合 true を返す
        template<typename F>
        static bool FindImpl(const std::shared_ptr<IMenuItem>& pTarget, Context<F>& context, const Aabb& crop) NN_NOEXCEPT
        {
            if(!pTarget)
            {
                return false;
            }

            // 無効なアイテムは無視する
            if(!pTarget->IsEnabled())
            {
                return false;
            }

            // 表示領域がない場合無視する
            auto scrop = crop.Intersect(Aabb::FromRectangle(MenuItemGeometry::GetGlobalRectangle(pTarget)));
            if(scrop.IsEmpty())
            {
                return false;
            }

            if(pTarget->IsFocusAcceptable())
            {
                if(context.pCandidate) // 2 つ目以降に見つけたもの
                {
                    NN_SDK_ASSERT_NOT_NULL(context.pCurrentFocused);
                    float distance = context.distanceFunction(pTarget, context.pCurrentFocused);

                    // より近ければ採用
                    if(distance < context.candidateDistance)
                    {
                        context.pCandidate = pTarget;
                        context.candidateDistance = distance;
                    }
                }
                else // 最初に見つけたもの
                {
                    context.pCandidate = pTarget;
                    if(context.pCurrentFocused)
                    {
                        context.candidateDistance = context.distanceFunction(pTarget, context.pCurrentFocused);
                    }
                    else
                    {
                        // pCurrentFocused == nullptr なら最初に見つけたものを採用
                        return true;
                    }
                }
            }

            // 子を調べる
            int n = pTarget->GetChildCount();
            for(int i = 0; i < n; i++)
            {
                if(FindImpl(pTarget->GetChild(i), context, scrop))
                {
                    return true;
                }
            }

            return false;
        }

    };

}}
