﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace ui{
    class IMenuItem;
    class IMenuPage;

    // UI 操作のハンドラに渡されるコンテキスト
    class MenuButtonHandleContext
    {
    public:
        MenuButtonHandleContext(
            const std::shared_ptr<IMenuItem>& pCurrentFocusedItem,
            const std::shared_ptr<IMenuItem>& pFirstHandlerItem,
            const std::shared_ptr<IMenuItem>& pPreviousHandlerItem,
            const std::shared_ptr<IMenuPage>& pPage
        ) NN_NOEXCEPT
            : m_pCurrentFocusedItem(pCurrentFocusedItem)
            , m_pFirstHandlerItem(pFirstHandlerItem)
            , m_pPreviousHandlerItem(pPreviousHandlerItem)
            , m_pPage(pPage)
        {
        }

        static MenuButtonHandleContext CreateNextContext(
            const std::shared_ptr<IMenuItem>& pPreviousHandlerItem,
            const MenuButtonHandleContext& sourceContext
        ) NN_NOEXCEPT
        {
            return MenuButtonHandleContext(
                sourceContext.m_pCurrentFocusedItem,
                sourceContext.m_pFirstHandlerItem,
                pPreviousHandlerItem,
                sourceContext.m_pPage
            );
        }

        std::shared_ptr<IMenuItem> GetCurrentFocusedItem() const NN_NOEXCEPT { return m_pCurrentFocusedItem; }
        std::shared_ptr<IMenuItem> GetFirstHandlerItem() const NN_NOEXCEPT { return m_pFirstHandlerItem; }
        std::shared_ptr<IMenuItem> GetPreviousHandlerItem() const NN_NOEXCEPT { return m_pPreviousHandlerItem; }
        std::shared_ptr<IMenuPage> GetPage() const NN_NOEXCEPT{ return m_pPage; }

    private:
        // 現在ページ内でフォーカスを持っているアイテム。
        std::shared_ptr<IMenuItem> m_pCurrentFocusedItem;

        // ユーザー入力のハンドリングを最初に行うアイテム。
        std::shared_ptr<IMenuItem> m_pFirstHandlerItem;

        // 前にハンドリングを行ったアイテム。
        // 今のアイテムが最初の場合 nullptr。
        std::shared_ptr<IMenuItem> m_pPreviousHandlerItem;

        // メニューページ。
        // すべてのハンドリング処理が終わってから反映される。
        std::shared_ptr<IMenuPage> m_pPage;
    };

    // NotifyFocusedItemChanged の引数に渡されるコンテキスト
    class MenuFocusedItemChangedContext
    {
    public:
        explicit MenuFocusedItemChangedContext(
            const std::shared_ptr<const IMenuItem>& pNewFocusedItem,
            const std::shared_ptr<const IMenuItem>& pOldFocusedItem,
            const std::shared_ptr<IMenuPage>& pPage
        ) NN_NOEXCEPT
            : m_pNewFocusedItem(pNewFocusedItem)
            , m_pOldFocusedItem(pOldFocusedItem)
            , m_pPage(pPage)
        {
        }

        std::shared_ptr<const IMenuItem> GetNewFocusedItem() const NN_NOEXCEPT { return m_pNewFocusedItem; }
        std::shared_ptr<const IMenuItem> GetOldFocusedItem() const NN_NOEXCEPT { return m_pOldFocusedItem; }
        std::shared_ptr<IMenuPage> GetPage() const NN_NOEXCEPT { return m_pPage; }

    private:
        // 新しくフォーカスを得たアイテム。
        // 誰もフォーカスを受け取らなくなった場合 nullptr になる。
        std::shared_ptr<const IMenuItem> m_pNewFocusedItem;

        // 直前までフォーカスを受け取っていたアイテム。
        // 誰もフォーカスを受け取っていなかったり、フォーカスを得ているアイテムが削除されていた場合 nullptr になる。
        std::shared_ptr<const IMenuItem> m_pOldFocusedItem;

        // メニューページ。
        std::shared_ptr<IMenuPage> m_pPage;
    };

    // UpdateLayout の引数に渡されるコンテキスト
    class MenuLayoutUpdateContext
    {
    public:
        explicit MenuLayoutUpdateContext(
            const std::shared_ptr<IMenuItem>& pCurrentFocusedItem
        ) NN_NOEXCEPT
            : m_pCurrentFocusedItem(pCurrentFocusedItem)
        {
        }

        std::shared_ptr<IMenuItem> GetCurrentFocusedItem() const NN_NOEXCEPT { return m_pCurrentFocusedItem; }

    private:
        // 現在ページ内でフォーカスを持っているアイテム。
        std::shared_ptr<IMenuItem> m_pCurrentFocusedItem;
    };


}
