﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemRadioButton.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Assert.h>
#include "../Config.h"
#include "../framework/Framework.h"
#include "util/ui_CalculateItemSize.h"
#include "util/ui_CalculateTextSize.h"
#include "util/ui_CalculateAlignedPosition.h"

namespace ui{

    MenuItemRadioButton::MenuItemRadioButton(
        const MenuItemRadioButtonStyle& style,
        std::shared_ptr<RadioButtonGroup>& group,
        const std::string& text
    ) NN_NOEXCEPT
    {
        m_pPanel = std::make_shared<panel::PanelText>();
        NN_ASSERT_NOT_NULL(m_pPanel);
        m_pPanel->SetText(text);

        m_pRadioButton = std::make_shared<RadioButton>(group);
        NN_ASSERT_NOT_NULL(m_pRadioButton);
        m_pRadioButton->SetValueChangedCallbackFunction(
            [this](bool value, const std::shared_ptr<IMenuPage>& pPage){ this->NotifyValueChangedImpl(value, pPage); }
        );

        SetStyleImpl(style);
    }

    MenuItemRadioButton::~MenuItemRadioButton() NN_NOEXCEPT
    {
        m_pRadioButton->SetValueChangedCallbackFunction({});
    }

    std::shared_ptr<MenuItemRadioButton> MenuItemRadioButton::SetValueAccessor(const std::function<bool ()>& getter, const std::function<void (bool)>& setter) NN_NOEXCEPT
    {
        m_ValueLoadFunction = getter;
        m_ValueStoreFunction = setter;
        return this->shared_from_this();
    }

    std::shared_ptr<MenuItemRadioButton> MenuItemRadioButton::SetValueChangedCallbackFunction(const std::function<void (bool, const std::shared_ptr<IMenuPage>&)>& f) NN_NOEXCEPT
    {
        m_ValueChangedCallbackFunction = f;
        return this->shared_from_this();
    }

    bool MenuItemRadioButton::IsSelected() const NN_NOEXCEPT
    {
        return m_SelectionState == SelectionState_Selected;
    }

    void MenuItemRadioButton::SetSelectedImpl(bool value, const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        // 大変面倒なことに以下のように呼び出される
        // MenuItemRadioButton::SetSelectedImpl()
        // -> RadioButton::SetSelected()
        // -> RadioButtonGroup::SetSelectedItem()
        // <- RadioButton::NotifySelectedStateChanged()
        // <- MenuItemRadioButton::NotifyValueChangedImpl()
        m_pRadioButton->SetSelected(value, pPage);

        // SetSelected を呼んだにも関わらず Unspecified のままだったら NotSelected に確定する
        if(m_SelectionState == SelectionState_Unspecified)
        {
            NotifyValueChangedImpl(false, pPage);
        }
    }

    void MenuItemRadioButton::NotifyValueChangedImpl(bool value, const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        auto prev = m_SelectionState;
        m_SelectionState = value ? SelectionState_Selected : SelectionState_NotSelected;

        if(prev != m_SelectionState && m_ValueChangedCallbackFunction)
        {
            m_ValueChangedCallbackFunction(value, pPage);
        }
    }

    bool MenuItemRadioButton::IsFocusAcceptable() const NN_NOEXCEPT
    {
        return true;
    }

    bool MenuItemRadioButton::IsTouchAcceptable() const NN_NOEXCEPT
    {
        return true;
    }

    MenuButtonResult MenuItemRadioButton::HandlePositiveButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT
    {
        SetSelectedImpl(true, context.GetPage());
        return MenuButtonResult::GetHandled();
    }

    MenuButtonResult MenuItemRadioButton::HandleTouched(const MenuButtonHandleContext& context, const nn::util::Vector3f&) NN_NOEXCEPT
    {
        context.GetPage()->ChangeFocus(this->shared_from_this());
        SetSelectedImpl(true, context.GetPage());
        return MenuButtonResult::GetHandled();
    }

    void MenuItemRadioButton::LoadValue(const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT
    {
        if(m_ValueLoadFunction)
        {
            SetSelectedImpl(m_ValueLoadFunction(), pPage);
        }
    }

    void MenuItemRadioButton::StoreValue() const NN_NOEXCEPT
    {
        if(m_ValueStoreFunction)
        {
            m_ValueStoreFunction(IsSelected());
        }
    }

    std::shared_ptr<panel::IPanel> MenuItemRadioButton::GetPanel() NN_NOEXCEPT
    {
        return m_pPanel;
    }

    //------------------------

    void MenuItemRadioButton::SetStyleImpl(const MenuItemRadioButtonStyle& style) NN_NOEXCEPT
    {
        m_Style = style;
    }

    void MenuItemRadioButton::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        const auto& text = m_pPanel->GetText();
        auto margin = m_Style.GetMargin();
        Size textSize = util::CalculateTextSize::Calculate(text.c_str(), m_Style.GetTextStyle(IsEnabled(), IsFocused(), IsSelected()).GetFontSize(), framework::TextWriterUsage_Text);
        Size layoutSize = {
            textSize.width + margin.left + margin.right,
            textSize.height + margin.top + margin.bottom
        };

        auto outerSize = util::CalculateItemSize::Calculate([&](){ return layoutSize; }, m_Style.GetSizeStyle());

        auto layoutPosition = util::CalculateAlignedPosition::Calculate(layoutSize, outerSize, m_Style.GetAlignment());

        SetLayoutSize(outerSize);
        m_TextPosition = {layoutPosition.x + margin.left, layoutPosition.y + margin.top};
    }

    void MenuItemRadioButton::UpdatePanel() NN_NOEXCEPT
    {
        m_pPanel->SetVisibility(panel::PanelVisibility::Visible);

        auto pos = this->GetLayoutPosition();
        auto size = this->GetLayoutSize();
        m_pPanel->SetPosition(pos.x, pos.y);
        m_pPanel->SetSize(size.width, size.height);

        auto bgColor = m_Style.GetBackgroundColor(IsEnabled(), IsFocused(), IsSelected());
        auto textStyle = m_Style.GetTextStyle(IsEnabled(), IsFocused(), IsSelected());

        m_pPanel->SetColor(bgColor);
        m_pPanel->SetTextPosition(m_TextPosition.x, m_TextPosition.y);
        m_pPanel->SetTextColor(textStyle.GetFontColor());
        m_pPanel->SetTextSize(textStyle.GetFontSize());
    }
}

