﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <vector>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../panel/panel_IPanel.h"
#include "ui_Types.h"
#include "ui_ItemAlignment.h"
#include "ui_MenuUpdateContext.h"

namespace ui{
    class IMenuPage;
    class IMenuItem;

    enum MenuButtonResultCode
    {
        // このアイテムではハンドルしなかったことを示す。
        // 親アイテムがハンドルする。
        MenuButtonResultCode_NotHandled = 0,

        // このアイテムでハンドルしたことを示す。
        // 親アイテムは何もしない。
        MenuButtonResultCode_Handled,

        MenuButtonResultCode_FocusChanging,
    };

    class MenuButtonResult
    {
    public:
        static MenuButtonResult GetNotHandled() NN_NOEXCEPT
        {
            MenuButtonResult value = {};
            value.m_Code = MenuButtonResultCode_NotHandled;
            return value;
        }

        static MenuButtonResult GetHandled() NN_NOEXCEPT
        {
            MenuButtonResult value = {};
            value.m_Code = MenuButtonResultCode_Handled;
            return value;
        }

        MenuButtonResultCode GetCode() const NN_NOEXCEPT
        {
            return m_Code;
        }

        bool IsHandled() const NN_NOEXCEPT
        {
            return m_Code != MenuButtonResultCode_NotHandled;
        }

    private:
        MenuButtonResultCode m_Code;
    };

    struct MenuItemParentInfo
    {
        std::weak_ptr<IMenuItem> pParent = {};
        int index = 0;
    };

    class IMenuItem
    {
    public:
        virtual ~IMenuItem() NN_NOEXCEPT
        {
        };

    //@name Structure
    //@{

        // このアイテムの名前を取得する。
        virtual std::string GetName() const NN_NOEXCEPT = 0;

        // このアイテムの名前を設定する。
        virtual void SetName(const std::string& value) NN_NOEXCEPT = 0;

        // 親アイテムが変更されたときに呼ばれる。
        virtual void OnParentItemChanged(const std::shared_ptr<IMenuItem>& pNewParent, size_t indexOnParent) NN_NOEXCEPT = 0;

        // 現在の親の情報を取得する。
        virtual MenuItemParentInfo GetParentInfo() const NN_NOEXCEPT = 0;

        // このアイテムの子のアイテムの数を取得する。子を持たないなら 0 を返す。
        virtual int GetChildCount() const NN_NOEXCEPT = 0;

        // 指定したインデックスの子を取得する。
        // 範囲外のインデックスを指定した場合 nullptr が返る。
        virtual std::shared_ptr<IMenuItem> GetChild(int index) NN_NOEXCEPT = 0;

    //@}

    //@name Geometry
    //@{

        // このアイテムの親アイテムの座標系での位置を取得する。
        // 親がない場合はスクリーン座標での位置を取得する。
        // この値は UpdateLayout の呼出で更新される。
        virtual Position GetLayoutPosition() const NN_NOEXCEPT = 0;

        // このアイテムの親アイテムの座標系での位置を設定する。
        // 通常は親の UpdateLayout の呼び出しにより設定されるためこの関数を明示的に呼び出す必要はない。
        virtual void SetLayoutPosition(const Position& value) NN_NOEXCEPT = 0;

        // このアイテムの大きさを取得する。
        // この値は UpdateLayout の呼出で更新される。
        virtual Size GetLayoutSize() const NN_NOEXCEPT = 0;

        // このアイテムと子孫のレイアウトを更新する。
        // この関数呼び出しで更新される可能性があるのは以下のもの
        //
        // - このアイテムの Size
        // - 子孫アイテムの Position と Size
        //
        // このアイテムの Position は更新されない。
        //
        // TORIAEZU:
        //   諸般の事情によりこの関数だけメンバ関数による再帰呼び出しを行う。
        //   TableCell の Update のせい。
        virtual void UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT = 0;

    //@}

    //@name Enablity
    //@{

        // このアイテムが有効かを取得する。
        // - 無効なアイテムは IsFocusAcceptable() に依らずフォーカスを受け取る対象から除外される。
        // - 無効なアイテムは IsTouchAcceptable() に依らずタッチを受け取る対象から除外される。
        virtual bool IsEnabled() const NN_NOEXCEPT = 0;

        // このアイテムの有効状態を変更する。
        virtual void SetEnablity(bool isEnabled) NN_NOEXCEPT = 0;

    //@}

    //@name Focus
    //
    // フォーカスの変更時には以下の順で呼び出される
    // (1) pOld->OnOutFocus();
    // (2) pNew->OnInFocus();
    // (3) pRoot->NotifyFocusedItemChanged();
    //@{

        // このアイテム自身がフォーカスを取得できるかを取得する。
        virtual bool IsFocusAcceptable() const NN_NOEXCEPT = 0;

        // このアイテムにフォーカスが移動するときに呼ばれる。
        virtual void OnInFocus() NN_NOEXCEPT = 0;

        // このアイテムがフォーカスを失うときに呼ばれる。
        virtual void OnOutFocus() NN_NOEXCEPT = 0;

        // ページ中のフォーカスを受け取るアイテムが変化したときに呼ばれる。
        // この関数ではこのアイテム単体の処理を行う。ツリー全体に通知する場合には util::ProcessNotifyFocusedItemChanged を使う。
        virtual void NotifyFocusedItemChanged(const MenuFocusedItemChangedContext& context) NN_NOEXCEPT = 0;

    //@}

    //@name Touch
    //@{

        // このアイテム自身がタッチ入力を受け付けるかを取得する。
        virtual bool IsTouchAcceptable() const NN_NOEXCEPT = 0;

    //@}

    //@name User Input
    //@{

        // いわゆる A ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandlePositiveButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // いわゆる B ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandleNegativeButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // いわゆる↑ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandleUpButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // いわゆる↓ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandleDownButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // いわゆる←ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandleLeftButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // いわゆる→ボタン押し。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessButtonHandling を使う。
        virtual MenuButtonResult HandleRightButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT = 0;

        // 左アナログスティック入力。
        // フォーカスを受け取っているアイテムに対して毎フレームこの関数が呼ばれる。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessAnalogStickHandling を使う。
        virtual MenuButtonResult HandleLeftAnalogStickInput(const MenuButtonHandleContext& context, const nn::util::Vector3f& value)  NN_NOEXCEPT = 0;

        // 右アナログスティック入力。
        // フォーカスを受け取っているアイテムに対して毎フレームこの関数が呼ばれる。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessAnalogStickHandling を使う。
        virtual MenuButtonResult HandleRightAnalogStickInput(const MenuButtonHandleContext& context, const nn::util::Vector3f& value)  NN_NOEXCEPT = 0;

        // タッチ入力。
        // 以下いずれかの条件で通知される
        //
        // - タッチの開始と終了がともにこのアイテムの上だった場合
        // - 何もない場所でタッチ開始した後、このアイテムの上でタッチ終了した場合
        //
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessTouchHandling を使う。
        virtual MenuButtonResult HandleTouched(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT = 0;

        // タッチが開始したことをハンドリングする。
        // タッチの終了は Up または Canceled により通知される。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessTouchHandling を使う。
        // @param[in] position スクリーン座標系におけるタッチ開始位置
        virtual MenuButtonResult HandleTouchDown(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT = 0;

        // タッチが離された。タッチ点がアイテムの外に出ても通知される。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessTouchHandling を使う。
        // @param[in] position スクリーン座標系におけるタッチ終了位置
        virtual MenuButtonResult HandleTouchUp(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT = 0;

        // タッチしながら動いた。タッチ点がアイテムの外に出ても通知される。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessTouchHandling を使う。
        // @param[in] position スクリーン座標系における現在のタッチ位置。
        virtual MenuButtonResult HandleTouchMove(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT = 0;

        // メニューの開閉などによりタッチ状態の追跡が終了した。
        // この関数ではこのアイテム単体の処理を行う。ツリーとして入力を処理する場合には util::ProcessTouchHandling を使う。
        virtual MenuButtonResult HandleTouchCanceled() NN_NOEXCEPT = 0;

    //@}

    //@name Load/Store Value
    //@{

        // このアイテムに対して外部から値を読み込む。
        // この関数ではこのアイテム単体の処理を行う。ツリー全体に対して呼び出す場合には util::ProcessLoadStoreValue を使う。
        virtual void LoadValue(const std::shared_ptr<IMenuPage>& pPage) NN_NOEXCEPT = 0;

        // このアイテムの値を外部に保存する。
        // この関数ではこのアイテム単体の処理を行う。ツリー全体に対して呼び出す場合には util::ProcessLoadStoreValue を使う。
        virtual void StoreValue() const NN_NOEXCEPT = 0;

    //@}

    //@name Rendering
    //@{

        // このアイテムの描画用オブジェクトを取得する。
        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT = 0;

        // このアイテムの描画用オブジェクトを更新する。
        // この関数ではこのアイテム単体の処理を行う。ツリー全体を更新する場合には util::ProcessUpdatePanel を使う。
        virtual void UpdatePanel() NN_NOEXCEPT = 0;

    //@}
    };

}
