﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneOverlay.h"

#include <sstream>

#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include "../../Config.h"
#include "../../config/NotificationParameter.h"
#include "../../framework/Framework.h"
#include "../../framework/Audio.h"

#include <nn/ovln/ovln_FormatForOverlay.h>
#include "ConvertControllerInfo.h"

#include "NotificationAccount.h"
#include "NotificationBatteryCharge.h"
#include "NotificationAudioVolume.h"
#include "NotificationScreenShot.h"
#include "NotificationMovie.h"
#include "NotificationHomeProhibited.h"
#include "NotificationSnapShotDumperProgress.h"

#include "NotificationLowBattery.h"
#include "NotificationControllerPairing.h"
#include "NotificationControllerBattery.h"
#include "NotificationPlayTimer.h"
#include "NotificationParentalControl.h"
#include "NotificationDownload.h"
#include "NotificationFriend.h"
#include "NotificationServerMaintenance.h"


#define NN_DEVOVL_GET_OVLN_DATA_EX(type,var,pSrcVar,typeStr) \
    NN_DEVOVL_DISP_LOG_NMSG("Message: " typeStr "\n");    \
    type var;                \
    if(pSrcVar->dataSize != sizeof(var))                \
    {                                                   \
        NN_DEVOVL_DISP_LOG_NMSG("[Error] invalid message data size\n"); \
        return nullptr;                                 \
    }                                                   \
    std::memcpy(&var, &pSrcVar->data, sizeof(var));

#define NN_DEVOVL_GET_OVLN_DATA(type,var,pSrcVar) \
    NN_DEVOVL_GET_OVLN_DATA_EX(nn::ovln::format::type ## Data, var, pSrcVar, #type)

#define NN_DEVOVL_SETUP_NOTIFICATION(expr)  if(!(expr)) { return nullptr; } else {}

namespace scene{ namespace overlay{

    namespace{

        // 充電状態通知
        std::shared_ptr<NotificationBatteryCharge> CreateNotificationBatteryCharge(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(BatteryStateChange, data, msg);

            auto pNotification = std::make_shared<NotificationBatteryCharge>();
            switch(nn::ovln::format::GetChargerType(data))
            {
            case nn::ovln::format::ChargerType_Unconnected:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupChargeStop(nn::ovln::format::GetBatteryChargePercentage(data)));
                break;
            case nn::ovln::format::ChargerType_EnoughPower:
            case nn::ovln::format::ChargerType_LowPower:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupChargeStart(nn::ovln::format::GetBatteryChargePercentage(data)));
                break;
            case nn::ovln::format::ChargerType_NotSupported:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupChargeError(nn::ovln::format::GetBatteryChargePercentage(data)));
                break;
            default:
                NN_DEVOVL_DISP_LOG_NMSG("unknown charger type(%d)\n", static_cast<int>(nn::ovln::format::GetChargerType(data)));
                return nullptr;
            }

            return pNotification;

        }

        // 音量変化通知
        std::shared_ptr<NotificationAudioVolume> CreateNotificationAudioVolume(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(AudioVolumeChanged, data, msg);

            NotificationAudioVolume::Target target;
            switch(data.audioOutputTarget)
            {
            case nn::ovln::format::AudioOutputTarget_HeadphoneMicJack:
                target = NotificationAudioVolume::Target_Headphone;
                break;
            case nn::ovln::format::AudioOutputTarget_Speaker:
                target = NotificationAudioVolume::Target_Speaker;
                break;
            case nn::ovln::format::AudioOutputTarget_Hdmi:
                target = NotificationAudioVolume::Target_Hdmi;
                break;
            case nn::ovln::format::AudioOutputTarget_UsbOutputDevice:
                target = NotificationAudioVolume::Target_UsbOutputDevice;
                break;
            default:
                target = NotificationAudioVolume::Target_Unknown;
                break;
            }

            float volume = static_cast<float>(data.volume) / 15.f;

            auto pNotification = std::make_shared<NotificationAudioVolume>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupVolume(target, volume, data.mute));
            return pNotification;
        }

        // 画面写真撮影通知
        std::shared_ptr<NotificationScreenShot> CreateNotificationScreenShot(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(ScreenShotCaptured, data, msg);

            auto pNotification = std::make_shared<NotificationScreenShot>();
            switch(data.result)
            {
            case nn::ovln::format::ScreenShotResult::Success:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupSuccess());
                break;
            case nn::ovln::format::ScreenShotResult::Failure:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupFailure());
                break;
            case nn::ovln::format::ScreenShotResult::NotPermitted:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupProhibited());
                break;
            case nn::ovln::format::ScreenShotResult::AlbumIsFull:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupAlbumIsFull());
                break;
            case nn::ovln::format::ScreenShotResult::AlbumFileCountLimit:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupAlbumFileCountLimit());
                break;
            case nn::ovln::format::ScreenShotResult::ThumbnailIsReady:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupThumbnailIsReady());
                break;
            case nn::ovln::format::ScreenShotResult::AlbumFileSystemError:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupFileSystemError());
                break;
            case nn::ovln::format::ScreenShotResult::ScreenCaptureError:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupScreenCaptureError());
                break;
            default:
                NN_DEVOVL_DISP_LOG_NMSG("unknown result(%d)\n", static_cast<int>(data.result));
                return nullptr;
            }

            return pNotification;
        }

        // 動画撮影通知
        std::shared_ptr<NotificationMovie> CreateNotificationMovie(uint32_t tag, const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            auto pNotification = std::make_shared<NotificationMovie>();

            switch(tag)
            {
            // ConrinuousRecodingData がある
            case nn::ovln::format::ContinuousRecordingStartedDataTag:
            case nn::ovln::format::ContinuousRecordingFlushFailedDataTag:
            case nn::ovln::format::ContinuousRecordingFlushSuccessDataTag:
                {
                    NN_DEVOVL_GET_OVLN_DATA(ContinuousRecording, data, msg);
                    switch(tag)
                    {
                    case nn::ovln::format::ContinuousRecordingStartedDataTag:
                        NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupStarted());
                        break;
                    case nn::ovln::format::ContinuousRecordingFlushSuccessDataTag:
                        NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupSuccess());
                        break;
                    case nn::ovln::format::ContinuousRecordingFlushFailedDataTag:
                        switch(data.result)
                        {
                        case nn::ovln::format::ContinuousRecordingResult::AlbumIsFull:
                            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupAlbumIsFull());
                            break;
                        case nn::ovln::format::ContinuousRecordingResult::AlbumFileCountLimit:
                            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupAlbumFileCountLimit());
                            break;
                        case nn::ovln::format::ContinuousRecordingResult::AlbumFileSystemError:
                            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupFileSystemError());
                            break;
                        default:
                            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupError());
                        }
                        break;
                    default:
                        NN_DEVOVL_DISP_LOG_NMSG("unknown result(%d-%d)\n", tag, static_cast<int>(data.result));
                        return nullptr;
                    }
                    break;
                }
            // ConrinuousRecodingData がない
            default:
                {
                    switch(tag)
                    {
                    case nn::ovln::format::ContinuousRecordingNotEnabledDataTag:
                        NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupNotEnabled());
                        break;
                    case nn::ovln::format::ContinuousRecordingNotPermittedDataTag:
                        NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupNotPermitted());
                        break;
                    case nn::ovln::format::ContinuousRecordingErrorDataTag:
                        NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupError());
                        break;
                    default:
                        NN_DEVOVL_DISP_LOG_NMSG("unknown result(%d)\n", tag);
                        return nullptr;
                    }
                }
            }

            return pNotification;
        }

        // Home 禁止通知
        std::shared_ptr<NotificationHomeProhibited> CreateNotificationHomeProhibited(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_UNUSED(msg);

            auto pNotification = std::make_shared<NotificationHomeProhibited>(msg->tag == nn::ovln::format::HomeButtonLongPressedIsNotPermittedDataTag);
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupHomeProhibited());
            return pNotification;
        }

        // 本体バッテリー僅少
        std::shared_ptr<NotificationLowBattery> CreateNotificationLowBattery(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(BatteryStateChange, data, msg);

            auto pNotification = std::make_shared<NotificationLowBattery>();
            int batteryPercent = nn::ovln::format::GetBatteryChargePercentage(data);

            if(batteryPercent <= config::NotificationConstant::LowBatteryThresholdPercent3)
            {
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupLowBattery3(batteryPercent));
            }
            else if(batteryPercent <= config::NotificationConstant::LowBatteryThresholdPercent2)
            {
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupLowBattery2(batteryPercent));
            }
            else if(batteryPercent <= config::NotificationConstant::LowBatteryThresholdPercent1)
            {
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupLowBattery1(batteryPercent));
            }
            else
            {
                // 残量が多い場合には無視する
                return nullptr;
            }

            return pNotification;
        }

        // コントローラ登録完了
        std::shared_ptr<NotificationControllerPairing> CreateNotificationControllerPairing(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(ControllerPairingCompleted, data, msg);

            auto pNotification = std::make_shared<NotificationControllerPairing>();
            switch(data.pairingMethod)
            {
            case nn::ovln::format::ControllerPairingMethod_Joint:
            case nn::ovln::format::ControllerPairingMethod_USB:
            case nn::ovln::format::ControllerPairingMethod_Button:
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->Setup(data.deviceType));
                break;
            default:
                NN_DEVOVL_DISP_LOG_NMSG("[Error] unknown pairing method\n");
                return nullptr;
            }

            return pNotification;
        }

        // コントローラバッテリー低下
        std::shared_ptr<NotificationControllerBattery> CreateNotificationControllerLowBattery(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(ControllerBatteryLevelLowEx, data, msg);

            auto controllerType = ConvertControllerInfo::Convert(data.controllerInfo);

            auto pNotification = std::make_shared<NotificationControllerBattery>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupLowBattery(controllerType));
            return pNotification;
        }

        // コントローラバッテリー切れ
        std::shared_ptr<NotificationControllerBattery> CreateNotificationControllerNoBattery(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(ControllerDisconnection, data, msg);

            if(data.reason != nn::ovln::format::ControllerDisconnectionReason_Battery)
            {
                NN_DEVOVL_DISP_LOG_NMSG("unknown disconnection reason(%d)\n", static_cast<int>(data.reason));
                return nullptr;
            }

            auto controllerType = ConvertControllerInfo::Convert(data.controllerInfo);

            auto pNotification = std::make_shared<NotificationControllerBattery>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupNoBattery(controllerType));
            return pNotification;
        }

        // プレイタイマー
        std::shared_ptr<NotificationPlayTimer> CreateNotificationPlayTimer(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(PctlRemainingTimeMessage, data, msg);

            auto pNotification = std::make_shared<NotificationPlayTimer>();
            if(data.remainingTime == 0)
            {
                switch(data.playTimerMode)
                {
                case nn::ovln::format::PctlPlayTimerModeFlag_Alarm:
                    NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupZeroPlayTimerAlarm());
                    break;
                case nn::ovln::format::PctlPlayTimerModeFlag_Suspend:
                    NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupZeroPlayTimerStop());
                    break;
                default:
                    return nullptr;
                }
            }
            else if(data.remainingTime > 0)
            {
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupLowPlayTimer(data.remainingTime));
            }
            else // if(data.remainingTime < 0)
            {
                NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupExceededPlayTimer(-data.remainingTime));
            }
            return pNotification;
        }

        // 見守り設定
        std::shared_ptr<NotificationParentalControl> CreateNotificationParentalControl(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_UNUSED(msg);

            auto pNotification = std::make_shared<NotificationParentalControl>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupChanged());
            return pNotification;
        }

        // ダウンロード
        std::shared_ptr<NotificationDownload> CreateNotificationDownloadComplete(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(DownloadComplete, data, msg);

            auto pNotification = std::make_shared<NotificationDownload>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupComplete(data.downloadedDataTypes, data.downloadedApplicationAttributes));
            return pNotification;
        }

        // コンテンツ利用可能
        std::shared_ptr<NotificationContentAvailable> CreateNotificationContentAvailable(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(ContentAvailable, data, msg);

            auto pNotification = std::make_shared<NotificationContentAvailable>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupComplete(data.availableDataType));
            return pNotification;
        }

        // フレンド
        std::shared_ptr<NotificationFriend> CreateNotificationFriend(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA_EX(nn::ovln::format::ListSummary,data,msg,"FriendListSummary");

            if(data.onlineFriendCount == 0 && data.unreadFriendRequestCount == 0)
            {
                return nullptr;
            }

            auto pNotification = std::make_shared<NotificationFriend>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupFriend(data.onlineFriendCount, data.unreadFriendRequestCount));
            return pNotification;
        }

        // サーバーメンテナンス予告
        std::shared_ptr<NotificationServerMaintenance> CreateNotificationServerMaintenanceScheduled(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_UNUSED(msg);

            auto pNotification = std::make_shared<NotificationServerMaintenance>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupScheduled());
            return pNotification;
        }

        // サーバーメンテナンス開始
        std::shared_ptr<NotificationServerMaintenance> CreateNotificationServerMaintenanceStart(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_UNUSED(msg);

            auto pNotification = std::make_shared<NotificationServerMaintenance>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupStart());
            return pNotification;
        }

        // アカウント
        std::shared_ptr<NotificationAccount> CreateNotificationAccount(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA(AccountUserStateChange, data, msg);

            auto pNotification = std::make_shared<NotificationAccount>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->Setup(data));
            return pNotification;
        }

        // SnapShotDumper
        std::shared_ptr<NotificationSnapShotDumperProgress> CreateNotificationSnapShotDumperProgress(const std::shared_ptr<nn::ovln::Message>& msg) NN_NOEXCEPT
        {
            NN_DEVOVL_GET_OVLN_DATA_EX(nn::ovln::format::DumpProgress, percentage, msg, "SnapShotDumperProgress");

            auto pNotification = std::make_shared<NotificationSnapShotDumperProgress>();
            NN_DEVOVL_SETUP_NOTIFICATION(pNotification->SetupProgress(percentage.dumpPercentage));
            return pNotification;
        }
    }

    //------------------------------------------------------------------------------

    SceneOverlay::SceneOverlay(const std::shared_ptr<void>& pArg) NN_NOEXCEPT
    {
        auto pParam = std::static_pointer_cast<SceneOverlayParameter>(pArg);
        NN_SDK_REQUIRES_NOT_NULL(pParam);

        m_pPanelContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelContainer->SetPosition(0, 0);
        m_pPanelContainer->SetSize(ScreenWidth, ScreenHeight);
        m_pPanelContainer->SetVisibility(panel::PanelVisibility::Transparent);
        m_pPanelContainer->SetPanelName("overlay");

        m_pOvlnChannel = pParam->GetOverlayNotificationChannel();
        NN_SDK_REQUIRES_NOT_NULL(m_pOvlnChannel);

        m_pSceneEventMultiWait = pParam->GetSceneEventMultiWait();
        NN_SDK_REQUIRES_NOT_NULL(m_pSceneEventMultiWait);

        m_pRunMode = pParam->GetRunMode();
        NN_SDK_REQUIRES_NOT_NULL(m_pRunMode);

        m_pPanelContainer->AddChild(m_ReactionNotificationSlot.GetPanel());
        m_pPanelContainer->AddChild(m_ConditionNotificationSlot.GetPanel());
    }

    int g_FrameCount = 0;
    //std::shared_ptr<PanelText> g_pPanelText;

    SceneUpdateResult SceneOverlay::Update() NN_NOEXCEPT
    {
        nn::os::Tick currentTick = nn::os::GetSystemTick();

        // 表示時間の満了をチェック
        m_ReactionNotificationSlot.CheckExpired(currentTick);
        m_ConditionNotificationSlot.CheckExpired(currentTick);

        // オーバーレイ通知を受信
        for(;;)
        {
            ThreadMessageType message;
            std::shared_ptr<ThreadMessageResultReporter> pReporter;
            if(!m_pOvlnChannel->TryReceiveMessage(&message, &pReporter))
            {
                break;
            }

            auto pUserData = std::static_pointer_cast<nn::ovln::Message>(pReporter->GetUserData());
            if(!pUserData)
            {
                continue;
            }

            //----------------------
            // リアクション通知
            //----------------------

            if(m_pRunMode->IsChargerStateChangeDisplayed() && message == nn::ovln::format::PowerManagementChargerStateChangeDataTag)
            {
                auto p = CreateNotificationBatteryCharge(pUserData);
                PushReactionNotification(p, currentTick);
            }

            if(m_pRunMode->IsAudioVolumeChangeDisplayed() && message == nn::ovln::format::AudioVolumeChangedDataTag)
            {
                auto p = CreateNotificationAudioVolume(pUserData);
                PushReactionNotification(p, currentTick);
            }

            if(m_pRunMode->IsScreenShotNotificationDisplayed() &&
                ( message == nn::ovln::format::ScreenShotSuccessDataTag
                || message == nn::ovln::format::ScreenShotFailureDataTag
                || message == nn::ovln::format::ScreenShotIsNotPermittedDataTag
                || message == nn::ovln::format::ScreenShotThumbnailIsReadyDataTag
                )
              )
            {
                auto p = CreateNotificationScreenShot(pUserData);
                PushReactionNotification(p, currentTick);
            }

            if(m_pRunMode->IsMovieNotificationDisplayed() &&
                ( message == nn::ovln::format::ContinuousRecordingStartedDataTag
                || message == nn::ovln::format::ContinuousRecordingNotEnabledDataTag
                || message == nn::ovln::format::ContinuousRecordingNotPermittedDataTag
                || message == nn::ovln::format::ContinuousRecordingErrorDataTag
                || message == nn::ovln::format::ContinuousRecordingFlushSuccessDataTag
                || message == nn::ovln::format::ContinuousRecordingFlushFailedDataTag
                )
              )
            {
                auto p = CreateNotificationMovie(message, pUserData);
                PushReactionNotification(p, currentTick);
            }

            if(m_pRunMode->IsHomeButtonProhibitionDisplayed() &&
                ( message == nn::ovln::format::HomeButtonIsNotPermittedDataTag
                || message == nn::ovln::format::HomeButtonLongPressedIsNotPermittedDataTag
                )
              )
            {
                auto p = CreateNotificationHomeProhibited(pUserData);
                PushReactionNotification(p, currentTick);
            }

            if(m_pRunMode->IsSnapShotDumperProgressDisplayed() && message == nn::ovln::format::DumpProgressTag)
            {
                auto p = CreateNotificationSnapShotDumperProgress(pUserData);
                PushReactionNotification(p, currentTick);
            }

            //----------------------
            // コンディション通知
            //----------------------

            if(m_pRunMode->IsLowBatteryNotificationDisplayed() && message == nn::ovln::format::PowerManagementLowBatteryDataTag)
            {
                auto p = CreateNotificationLowBattery(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsControllerPairingDisplayed() && message == nn::ovln::format::ControllerPairingCompletedDataTag)
            {
                auto p = CreateNotificationControllerPairing(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsControllerLowBatteryDisplayed() && message == nn::ovln::format::ControllerBatteryLevelLowExDataTag)
            {
                auto p = CreateNotificationControllerLowBattery(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsControllerDisconnectionDisplayed() && message == nn::ovln::format::ControllerDisconnectionDataTag)
            {
                auto p = CreateNotificationControllerNoBattery(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsPctlRemainingTimeDisplayed() && message == nn::ovln::format::PctlRemainingTimeMessage)
            {
                auto p = CreateNotificationPlayTimer(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsPctlSettingChangedDisplayed() && message == nn::ovln::format::PctlSettingChangedMessage)
            {
                auto p = CreateNotificationParentalControl(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsDownloadCompletionDisplayed() && message == nn::ovln::format::DownloadCompleteDataTag)
            {
                auto p = CreateNotificationDownloadComplete(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if (m_pRunMode->IsContentAvailableDisplayed() && message == nn::ovln::format::ContentAvailableDataTag)
            {
                auto p = CreateNotificationContentAvailable(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsFriendNotificationDisplayed() && message == nn::ovln::format::ListSummaryTag)
            {
                auto p = CreateNotificationFriend(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsServerMaintenanceScheduledDisplayed() && message == nn::ovln::format::ServerMaintenanceNoticeDataTag)
            {
                auto p = CreateNotificationServerMaintenanceScheduled(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsServerMaintenanceStartDisplayed() && message == nn::ovln::format::ServerMaintenanceStartDataTag)
            {
                auto p = CreateNotificationServerMaintenanceStart(pUserData);
                PushConditionNotification(p, currentTick);
            }

            if(m_pRunMode->IsAccountDisplayed() && message == nn::ovln::format::AccountUserStateChangeTag)
            {
                auto p = CreateNotificationAccount(pUserData);
                PushConditionNotification(p, currentTick);
            }

            // SE 再生

            if(m_pRunMode->IsNotificationSoundEffectEnabled())
            {
                // 通知に合わせて SE を再生する
                // (ダッキングは未実装なので無音を再生)
                Audio::Playback(Audio::SeId::Silence);
            }

        }

        // コンディション通知を設定
        PresentConditionNotification(currentTick);

        // スロットの位置を設定
        if(m_ConditionNotificationSlot.HasNotification())
        {
            m_ConditionNotificationSlot.SetNotificationSlotIndex(0);
            m_ReactionNotificationSlot.SetNotificationSlotIndex(1);
        }
        else
        {
            m_ReactionNotificationSlot.SetNotificationSlotIndex(0);
        }

        // 描画を更新
        m_ReactionNotificationSlot.UpdatePanel();
        m_ConditionNotificationSlot.UpdatePanel();

        return SceneUpdateResult::GetKeepResult();
    }// NOLINT(impl/function_size)

    std::shared_ptr<panel::IPanel> SceneOverlay::GetPanel() NN_NOEXCEPT
    {
        return m_pPanelContainer;
    }

    void SceneOverlay::PushReactionNotification(const std::shared_ptr<INotification>& pNotification, nn::os::Tick currentTick) NN_NOEXCEPT
    {
        if(!pNotification)
        {
            return;
        }
        pNotification->ReadyToAppear();
        m_ReactionNotificationSlot.SetNotification(pNotification, currentTick, m_pSceneEventMultiWait);
    }

    void SceneOverlay::PushConditionNotification(const std::shared_ptr<INotification>& pNotification, nn::os::Tick currentTick) NN_NOEXCEPT
    {
        NN_UNUSED(currentTick);
        if(!pNotification)
        {
            return;
        }
        if(m_ConditionNotificationQueue.size() < config::NotificationConstant::ConditionNotificationQueueSize)
        {
            m_ConditionNotificationQueue.push_back(pNotification);
        }
        else
        {
            NN_DEVOVL_DISP_LOG_NMSG("[Error] Condition notification queue is full\n");
        }
    }

    void SceneOverlay::PresentConditionNotification(nn::os::Tick currentTick) NN_NOEXCEPT
    {
        if(m_ConditionNotificationSlot.HasNotification())
        {
            return;
        }
        if(m_ConditionNotificationQueue.empty())
        {
            return;
        }

        auto itHighest = m_ConditionNotificationQueue.begin();
        auto priorityHighest = (*itHighest)->GetPriority();
        {
            auto it = itHighest;
            ++it;
            for(; it != m_ConditionNotificationQueue.end(); ++it)
            {
                auto& p = *it;
                auto priority = p->GetPriority();
                // 優先度が高い（値が小さい）ものを探す。
                // 優先度が同一の場合は先にエンキューされた方を使う。
                if(priority < priorityHighest)
                {
                    itHighest = it;
                    priorityHighest = priority;
                }
            }
        }

        NN_SDK_ASSERT(itHighest != m_ConditionNotificationQueue.end());
        auto pNotification = *itHighest;
        m_ConditionNotificationQueue.erase(itHighest);

        pNotification->ReadyToAppear();
        m_ConditionNotificationSlot.SetNotification(pNotification, currentTick, m_pSceneEventMultiWait);
        NN_DEVOVL_DISP_LOG_NMSG("Presenting condition notification (remaining %d)\n", static_cast<int>(m_ConditionNotificationQueue.size()));
    }

}}
