﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ovln/format/ovln_ControllerMessage.h>

namespace scene{ namespace overlay{

    enum ControllerType
    {
        ControllerType_Invalid,
        ControllerType_Unknown,
        ControllerType_FullKeyController,
        ControllerType_JoyDualLeft,
        ControllerType_JoyDualRight,
        ControllerType_JoyDualBoth,
        ControllerType_JoyConLeftVertical,
        ControllerType_JoyConLeftHorizontal,
        ControllerType_JoyConRightVertical,
        ControllerType_JoyConRightHorizontal,
        ControllerType_HandheldLeft,
        ControllerType_HandheldRight,
        ControllerType_HandheldBoth,
    };

    class ConvertControllerInfo
    {
    public:
        static ControllerType Convert(const nn::ovln::format::ControllerInfo& cInfo) NN_NOEXCEPT
        {
            // FullKeyコントローラ
            if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::FullKeyController>())
            {
                return ControllerType_FullKeyController;
            }
            // Dual持ち
            else if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() && cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
            {
                bool isLeft  = cInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft >();
                bool isRight = cInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>();
                if(isLeft && isRight)
                {
                    return ControllerType_JoyDualBoth;
                }
                else if(isLeft)
                {
                    return ControllerType_JoyDualLeft;
                }
                else if(isRight)
                {
                    return ControllerType_JoyDualRight;
                }
                else
                {
                    return ControllerType_Invalid;
                }
            }
            // 1 本持ち(L)
            else if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
            {
                switch(cInfo.GetJoyHoldType())
                {
                case nn::hid::NpadJoyHoldType_Vertical:
                    return ControllerType_JoyConLeftVertical;
                case nn::hid::NpadJoyHoldType_Horizontal:
                    return ControllerType_JoyConLeftHorizontal;
                default:
                    return ControllerType_Invalid;
                }
            }
            // 1 本持ち(R)
            else if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
            {
                switch(cInfo.GetJoyHoldType())
                {
                case nn::hid::NpadJoyHoldType_Vertical:
                    return ControllerType_JoyConRightVertical;
                case nn::hid::NpadJoyHoldType_Horizontal:
                    return ControllerType_JoyConRightHorizontal;
                default:
                    return ControllerType_Invalid;
                }
            }
            // 携帯機コントローラ
            else if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::Handheld>())
            {
                bool isLeft  = cInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft >();
                bool isRight = cInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>();
                if(isLeft && isRight)
                {
                    return ControllerType_HandheldBoth;
                }
                else if(isLeft)
                {
                    return ControllerType_HandheldLeft;
                }
                else if(isRight)
                {
                    return ControllerType_HandheldRight;
                }
                else
                {
                    return ControllerType_Invalid;
                }
            }
            // 未知
            else if(cInfo.deviceType.Test<nn::hid::system::NpadDeviceType::Unknown>())
            {
                return ControllerType_Unknown;
            }
            else
            {
                return ControllerType_Invalid;
            }
        }
    };

}}
